package main

/* webserver.go: contains web server handlers and methods. */

/*
The start to a web server built into glitch. Lets find some best practices
   and expand upon this.

   Ideas:
   - data presentation, such as log files and command histories
   - save links that are seen in chat and list them.
   - !seen bot (works in chat too), "so and so last chatted in #channel @ time"
   - slack web hooks
   - git web hooks
   - skadi/jenkins hooks
   - chat graphs per user/channel.
   - Turn g.Handlers into dynamic routes: "/handler/"+handler.Name()
      - Requires adding a http handler to the slack.Handlers interfacg.
*/

import (
	"expvar"
	"fmt"
	"log"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
)

// Dummy Health handler
// TODO: replace with github.com/trivago/tgo/thealthcheck/
func healthHandler(w http.ResponseWriter, r *http.Request) {
        w.WriteHeader(http.StatusOK)
}

// Creates HTTP interface for expvar. Called by initialize().
func (g *Glitch) createHTTPListener() {
	if !strings.Contains(g.Export.bindAddr.Value(), ":") {
		exp.Debug("HTTP Server is Disabled.")
		return
	}

	// Add new HTTP routes herg.
	g.Router.HandleFunc("/", g.httpRouteExports)

        // respond at url path '/health' with '200 ok' to satisfy
        // cops health check
        g.Router.HandleFunc("/health", healthHandler)

	g.HTTPServer = &http.Server{
		Addr: g.Export.bindAddr.Value(),
		// Good practice to set timeouts to avoid Slowloris attacks.
		WriteTimeout: time.Second * 15,
		ReadTimeout:  time.Second * 15,
		IdleTimeout:  time.Second * 60,
		Handler:      g.Router, // Pass our instance of gorilla/mux in.
	}
	go func() {
		log.Print("Debug Data available at http://", g.Export.bindAddr.Value(), "/debug/vars")
		// Sleep a couple second in case of reload and tcp port not being shut down.
		time.Sleep(2 * time.Second)
		// Open a web interface so we can provide data in json format using expvar.
		if err := g.HTTPServer.ListenAndServe(); err != nil {
			log.Println("HTTP Listener Stopped:", err)
		}
	}()
}

// This HTTP route displays the expvar variables. Currently bound to "/"
func (g *Glitch) httpRouteExports(w http.ResponseWriter, r *http.Request) {
	g.Export.httpVisits.Add(1)
	extraMsg := ""
	data, err := exp.MakePrettyJSONString(expvar.Get(BotName).String())
	if err != nil {
		extraMsg += "This JSON may have errors. Found: " + err.Error()
	}

	_, _ = fmt.Fprintln(w, "<html>\n <body>\n  <h1>"+BotName+"</h1>\n"+
		"  <h4>"+extraMsg+"</h4>\n  <pre>\n"+string(data)+"\n  </pre>\n"+
		"  <a href=\"/debug/vars\">JSON API</a>\n </body>\n</html>")
}
