package exports

/* This handler provides a way to view the expvar data from within Slack
   You also have here, a very good example of how to write a handler. */

import (
	// standard
	"expvar"
	"log"

	// external
	goslack "github.com/nlopes/slack"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// Init sets up the handler.
func Init(cfg *Config, slackClient slack.Client) slack.Handler {
	handler := &handler{slackClient: slackClient, cfg: cfg}
	// These must be initialized here or expvar breaks.
	handler.export.Init()
	handler.export.Set("call_counter", &handler.export.callCounter)
	handler.export.Set("error_counter", &handler.export.errCounter)
	log.Print("Exports Handler Loaded!")
	return handler
}

// IsMatch is called for every slack message the bot sees.
func (h *handler) IsMatch(msg slack.Message) bool {
	// Return true if the message is equal to our trigger command.
	return msg.Data.Text == "!expvar" || msg.Data.Text == "!debug"
}

// Execute is called if IsMatch returns True.
func (h *handler) Execute(msg slack.Message) error {
	switch msg.Data.Text {
	case "!expvar":
		log.Print("Processing '!expvar' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
		h.export.callCounter.Add(1)
		// All the magic happens in this following line of code.
		data, err := exp.MakePrettyJSONString(expvar.Get(h.cfg.BotName).String())
		if err != nil {
			h.export.errCounter.Add(1)
			return err
		}
		// Send a pretty json reply.
		for _, s := range exp.SplitSubN(string(data), 3990) {
			h.slackClient.SendReply(msg, "```\n"+s+"\n```")
		}
	case "!debug":
		log.Print("Processing '!debug' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
		for _, mapName := range exp.GetMapList() {
			// Create one attachment for each map.
			attachment := goslack.Attachment{Pretext: "Expvar Map: *" + mapName + "*"}
			mapString, err := exp.GetMapValuesString(mapName)
			if err != nil {
				return err
			}
			if len(mapString) > 0 { // don't print empty maps.
				for t, v := range mapString {
					// Create one field for each k/v pair.
					s := false
					if len(t) < 20 && len(v) < 20 {
						s = true // it's short, literally.
					}
					attachment.Fields = append(attachment.Fields, goslack.AttachmentField{Title: t, Value: v, Short: s})
				}
				h.slackClient.SendAttachments(msg.RespondTo, "", []goslack.Attachment{attachment})
			}
		}
	}

	return nil
}

// Export must return the expvar Map this handler uses to export data.
func (h *handler) Export() *expvar.Map {
	return &h.export.Map
}

// Name is currently only used for expvar display, but may have other uses.
func (h *handler) Name() string {
	return "exports"
}

// If the handler has any routines to stop, this should stop them.
func (h *handler) Stop() error {
	return nil
}
