package exports

import (
	// standard
	"expvar"
	"testing"

	// external
	"github.com/stretchr/testify/assert"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack/slackfakes"
)

// Add tests.

func TestInitAndIsMatch(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	slackClient := new(slackfakes.FakeClient)
	config := &Config{BotName: "glitch"}

	h := Init(config, slackClient)
	// Test all the things Init() does too.
	a.NotNil(h.Export())
	a.EqualValues("0", h.Export().Get("call_counter").String())
	a.EqualValues("0", h.Export().Get("error_counter").String())

	msg := slack.Message{}
	a.False(h.IsMatch(msg))
	msg.Data.Text = "!notforus"
	a.False(h.IsMatch(msg))
	msg.Data.Text = "hi guys! what's up! :bleedPurple"
	a.False(h.IsMatch(msg))
	msg.Data.Text = "!expvar"
	a.True(h.IsMatch(msg))
	msg.Data.Text = "!expvar "
	a.False(h.IsMatch(msg))
}

// Make sure Stop() doesn't produce vomit, somehow.
func TestStop(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	config := &Config{BotName: "glitch"}
	// Mock the data
	slackClient := new(slackfakes.FakeClient)
	h := Init(config, slackClient)
	a.Nil(h.Stop(), "Stop must not return an error")
}

func TestName(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	config := &Config{BotName: "glitch"}
	slackClient := new(slackfakes.FakeClient)
	h := Init(config, slackClient)
	a.EqualValues("exports", h.Name(), "package name is wrong?")
}

func TestExecute(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	config := &Config{BotName: "glitch"}
	slackClient := new(slackfakes.FakeClient)
	h := Init(config, slackClient)
	msg := slack.Message{}
	msg.Data.Text = "!expvar"

	// This allows us some control over the json going into the function.
	exs := expvar.NewString("the_key")
	exs.Set("theSetting")
	ex := expvar.NewMap("glitch")
	ex.Set("the_key", exs)

	// Providing a valid set of data, Execute it and inspect the results.
	err := h.Execute(msg)
	a.Nil(err)
	a.EqualValues("1", h.Export().Get("call_counter").String())
	// make sure all our slack calls were hit an appropriate number of times.
	a.EqualValues(2, slackClient.GetEntityNameCallCount(), "the channel should only be resolved twice")
	// make sure a correct reply was sent to the user.
	a.EqualValues(1, slackClient.SendReplyCallCount(), "there must be only one reply, this will probably panic now")
	replymsg, replysent := slackClient.SendReplyArgsForCall(0)
	a.EqualValues(msg.Data.Text, replymsg.Data.Text, "the wrong reply msg was used to reply to user")
	a.EqualValues("```\n{\n   \"the_key\": \"theSetting\"\n}\n```", replysent, "the wrong reply was sent to user")

	// now make sure invalid json produces an error, and our counters increment.
	var broke *expvar.Map
	// adding an unitialized nil map pointer to our map will produce invalid json (trick)
	ex.Set("broke", broke)
	err = h.Execute(msg)
	a.NotNil(err)
	a.EqualValues("2", h.Export().Get("call_counter").String())
	a.EqualValues("1", h.Export().Get("error_counter").String())
	a.EqualValues(4, slackClient.GetEntityNameCallCount(), "the channel should only be resolved twice")
	a.EqualValues(1, slackClient.SendReplyCallCount(), "there must be no replies on this test, counter incremented")
}
