package incidents

import (
	// standard
	"expvar"
	"log"
	"os"
	"strings"
	"time"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// Init is called during setup of all commands and is used to register the correct callbacks by returning a common.Command
func Init(config *Config, slackClient slack.Client, jiraClient jira.Client) slack.Handler {
	config.ChatLogDir = slackClient.ReturnLogDir("incidents")
	handler := &handler{
		cfg:             config,
		jiraClient:      jiraClient,
		slackClient:     slackClient,
		trackedChannels: make([]*trackedChannel, 0),
		stopChan:        make(chan bool),
	}
	handler.export.Init()
	handler.export.Set("osRemove_Errors", &handler.export.osRemoveErrors)
	handler.export.Set("osOpen_Errors", &handler.export.osOpenErrors)
	handler.export.Set("fileClose_Errors", &handler.export.fileCloseErrors)
	handler.export.Set("attachFile_Errors", &handler.export.attachFileErrors)

	if config.TransitionID == "" {
		config.TransitionID = defaultTransitionID
	}
	if config.ProjectKey == "" {
		config.ProjectKey = defaultProject
	}
	if config.IssueType == "" {
		config.IssueType = defaultIssueType
	}

	// Create the chat log dir if it doesn't exist
	if err := os.MkdirAll(config.ChatLogDir, 0755); err != nil {
		log.Println("Error creating chat log directory", err) // We should probably have Init() return an error so we can handle setup failures
	}

	// If the bot got restarted while it was tracking messages for a channel, we need to set up these tracked channels again
	// We can tell which channels we should be tracking by examining the filenames in the config.ChatLogDir directory
	// We also use this method to check JIRA to see if any incidents that we are traking have been closed manually on the
	// JIRA side rather than via the bot, if so, then we upload the log as if we'd called !close via the bot
	if err := handler.checkTrackedChannels(); err != nil {
		log.Println("Error setting up tracked channels", err)
	}

	// We're going to check things every 10 minutes in case someone has edited status in JIRA
	ticker := time.NewTicker(time.Minute * 10)

	go func() {
		for {
			select {
			case <-ticker.C:
				log.Println("Running handler.checkTrackedChannels().")
				if err := handler.checkTrackedChannels(); err != nil {
					log.Println("Error refreshing tracked channels", err)
				}
			case <-handler.stopChan:
				log.Println("Stopping Channel Tracker Routine")
				return
			}
		}
	}()
	log.Print("Incidents Handler Loaded! Jira Config: '", config.ProjectKey+"/"+config.IssueType, "' - Commands: !open !move !close")
	return handler
}

func (h *handler) Export() *expvar.Map {
	return &h.export.Map
}

func (h *handler) Name() string {
	return "incidents"
}

// Stop shuts off the channel tracker routine.
func (h *handler) Stop() error {
	h.stopChan <- true
	return nil
}

func (h *handler) IsMatch(msg slack.Message) bool {
	// If it's a recognised command or a tracked channel, return true.
	if isTracked, _ := h.isTrackedChannel(msg.Data.Channel); isTracked ||
		msg.Data.Text == "!open" || strings.HasPrefix(msg.Data.Text, "!open ") ||
		msg.Data.Text == "!close" || strings.HasPrefix(msg.Data.Text, "!close ") ||
		msg.Data.Text == "!move" || strings.HasPrefix(msg.Data.Text, "!move ") {
		return true
	}
	return false
}

func (h *handler) Execute(msg slack.Message) error {
	// Work out if we've been given a command we should respond to
	cmdArgs := ""
	matches := strings.SplitN(msg.Data.Text, " ", 2)
	if len(matches) > 1 {
		cmdArgs = matches[1]
	}
	switch isTracked, trackedChannel := h.isTrackedChannel(msg.Data.Channel); {
	case matches[0] == "!open":
		log.Print("Processing '!open ", cmdArgs, "' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
		return h.openIncident(msg, cmdArgs)
	case matches[0] == "!move":
		log.Print("Processing '!move ", cmdArgs, "' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
		return h.moveIncident(msg, cmdArgs)
	case matches[0] == "!close":
		log.Print("Processing '!close ", cmdArgs, "' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
		return h.closeIncident(msg, cmdArgs)
	case isTracked:
		trackedChannel.logMessage(msg)
	}

	return nil
}
