package incidents

import (
	// standard
	"log"
	"os"
	"path"
	"regexp"
	"strings"
	"time"

	// external
	"github.com/pkg/errors"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// openIncident starts tracking messages for a channel and creates a JIRA task.
func (h *handler) openIncident(msg slack.Message, description string) error {
	h.slackClient.SendReply(msg, "Creating incident")

	// If there was a description passed as part of the !open command then use it for the description and append to the title
	if description != "" {
		description = time.Now().UTC().Format("Incident: 2006-01-02 15:04 UTC") + " - " + description
	} else {
		description = time.Now().UTC().Format("Incident: 2006-01-02 15:04 UTC")
	}

	ticket := &jira.Ticket{
		ProjectKey:  h.cfg.ProjectKey,
		Summary:     description,
		Description: description,
		IssueType:   h.cfg.IssueType,
	}

	if err := h.jiraClient.CreateTicket(ticket); err != nil {
		return err
	}

	// Create the tracked channel and set up channel for processing incoming messages.
	trackedChannel, err := h.createTrackedChannel(msg.Data.Channel, ticket.TicketKey)
	if err != nil {
		return err
	}

	// We want to track the !open message too.
	trackedChannel.logMessage(msg)

	// Let the user know we've started tracking and return the ticket URL.
	log.Println("Opened new issue for incident:", ticket.TicketKey)
	h.slackClient.SendReply(msg, "Opened new issue for incident: "+ticket.TicketKey+". See: "+ticket.URL+".\nMessages are being logged for this incident.")

	return nil
}

// moveIncident changes the tracked channel for an incident.
func (h *handler) moveIncident(msg slack.Message, args string) error {
	msgParts := strings.SplitN(args, " ", 2)
	if args == "" || len(msgParts) != 2 {
		return errors.New("Not enough arguments given, use this format: `!move <issue> <channel>`")
	}

	// Make sure we've been given appropriate arguments
	ticketKey, channelID := msgParts[0], msgParts[1]
	if chanMatches := regexp.MustCompile(slack.ChannelRegex).FindStringSubmatch(channelID); len(chanMatches) == 2 && chanMatches[1] != "" {
		channelID = chanMatches[1]
		if regexp.MustCompile(numberRegex).MatchString(ticketKey) {
			ticketKey = h.cfg.ProjectKey + "-" + ticketKey
		}
	} else {
		return errors.New("Invalid arguments given, use this format: `!move <issue> <channel>`")
	}

	trackedChannel := h.getTrackedChannelByJiraKey(ticketKey)
	if trackedChannel == nil {
		return errors.New("I am not tracking an incident with ID " + ticketKey)
	} else if tcid := trackedChannel.channelID; tcid != msg.Data.Channel {
		return errors.New("This incident is currently being tracked in <#" + tcid + "> please move from there.")
	}

	// Record the !move message
	trackedChannel.logMessage(msg)
	// Update the tracked channel
	close(trackedChannel.messages)
	close(trackedChannel.close)

	oldPath := trackedChannel.getLogPath(h.cfg.ChatLogDir)
	trackedChannel.channelID = channelID // Change ID to get new path.
	// Move file to new path
	if err := os.Rename(oldPath, trackedChannel.getLogPath(h.cfg.ChatLogDir)); err != nil {
		// Do we need to do more cleanup here? Perhaps not.
		return err
	}

	fh, err := os.OpenFile(trackedChannel.getLogPath(h.cfg.ChatLogDir), os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		return err
	}
	// Reinit the listener/writer
	go trackedChannel.watchMessages(fh, h.slackClient)

	// Let the user know we've moved the incident
	h.slackClient.SendReply(msg, "Successfully moved logging to <#"+channelID+">, messages in that channel will be recorded.")
	// Send a message to the new channel that we are now logging messages
	h.slackClient.SendMessage(channelID, []string{}, "Now tracking incident "+ticketKey+" in this channel, messages will be recorded to Jira.")
	return nil
}

// closeIncident stops the channel tracker, attaches the log file and closes the ticket.
func (h *handler) closeIncident(msg slack.Message, ticketKey string) error {
	tickets, errJ := h.jiraClient.GetOpenTickets(h.cfg.ProjectKey)
	if errJ != nil {
		return errors.Wrap(errJ, "GetOpenTickets")
	}
	// No ticket key specified, so list them
	if ticketKey == "" {
		// If there's only 1 incident open, close it
		if len(tickets) == 1 {
			ticketKey = tickets[0].TicketKey
		} else if keys := ""; len(tickets) > 0 {
			for _, t := range tickets {
				keys = keys + " " + t.TicketKey
			}
			return errors.Errorf("There are %v open issues:%v.\nPick one with `!close <issue>`", len(tickets), keys)
		} else {
			return errors.New("There are no open incidents! Create one with `!open <description>`")
		}
	} else if regexp.MustCompile(numberRegex).MatchString(ticketKey) {
		// We've just been passed the number, prepend ProjectKey
		ticketKey = h.cfg.ProjectKey + "-" + ticketKey
	}

	var foundTicket *jira.Ticket
	for _, ticket := range tickets {
		if strings.EqualFold(ticket.TicketKey, ticketKey) {
			foundTicket = ticket
			break
		}
	}

	// We can't find an open ticket with this key, but we might still have a log for it if it's been closed on the JIRA side
	// So check the tracked channels and if present, grab the ticket directly
	if foundTicket == nil {
		if tc := h.getTrackedChannelByJiraKey(ticketKey); tc != nil {
			var err error
			if tc.incidentKey != ticketKey {
				return errors.New("no open incident with key " + ticketKey)
				// We found a tracked channel so set foundTicket accordingly
			} else if foundTicket, err = h.jiraClient.GetTicket(ticketKey); err != nil {
				return errors.Wrap(err, "GetTicket")
			}
		}
	}

	if foundTicket == nil {
		return errors.New("No open incident with key " + ticketKey)
	}

	/* We have a ticket, let's close it! */

	// We might want to close something that for some reason our tracking didn't pick up, if so, then check for a log file just in case
	// Chances are this won't exist but it can't hurt to check
	logPath := path.Join(h.cfg.ChatLogDir, msg.RespondTo+"-"+foundTicket.TicketKey+".log")

	// First check for a tracked channel, and close the file, if we're specifically tracking in a difderent channel then direct the user there
	if tc := h.getTrackedChannelByJiraKey(foundTicket.TicketKey); tc == nil {
		log.Println("No (tc) log found for open Jira Incident:", foundTicket.TicketKey)
	} else if msg.RespondTo != tc.channelID {
		return errors.New("This incident is currently being tracked in <#" + tc.channelID + "> please close from there")
	} else {
		// We want to track the !close message too
		tc.logMessage(msg)
		// Give the logger a moment to read the buffered channel before it's closed.
		time.Sleep(100 * time.Microsecond)
		// Set the correct log path.
		logPath = tc.getLogPath(h.cfg.ChatLogDir)
		// Remove the tracked channel from our internal running tally.
		h.deleteTrackedChannel(foundTicket.TicketKey)
	}

	/* Attach the file to Jira and Transiiton the Ticket. */

	// Turn the attachment portion of this into another method.
	attachmentStatusMessage := "The chat history for this incident has been attached to the ticket."
	file, err := os.Open(logPath)
	if err == nil {
		defer func() {
			if err = file.Close(); err != nil {
				log.Printf("Error closing file [%v] %v", logPath, err)
				h.export.fileCloseErrors.Add(1)
				return
			}
			if err = os.Remove(logPath); err != nil {
				log.Printf("Error removing file [%v] %v", logPath, err)
				h.export.osRemoveErrors.Add(1)
			}
		}()
	}
	if os.IsNotExist(err) {
		attachmentStatusMessage = "No log was found for the incident and nothing will be attached to the Jira ticket. Sorry about that. :sadpanda:"
	} else if err != nil {
		attachmentStatusMessage = "An error occured opening the chat log and nothing will be attached to the Jira ticket. Sorry about that. :sadpanda:"
		h.export.osOpenErrors.Add(1)
		log.Println("os.Open(logPath) returned a strange error:", err)
	} else if err = h.jiraClient.AttachFile(foundTicket.TicketKey, file, foundTicket.TicketKey+"-incident-log.log"); err != nil {
		attachmentStatusMessage = "There was a problem attaching this incident's chat history to the JIRA ticket."
		log.Println("jiraClient.AttachFile() returned a strange error:", err)
		h.export.attachFileErrors.Add(1)
	}

	// Close ticket (it's possible for it to already be closed if it was closed on the JIRA side, so don't do it if it's already closed)
	if foundTicket.Status != "Closed" {
		if err := h.jiraClient.TransitionTicket(foundTicket.TicketKey, h.cfg.TransitionID); err != nil {
			attachmentStatusMessage += " There was a problem closing the Jira ticket: " + err.Error()
		}
	}

	h.slackClient.SendReply(msg, "Issue "+foundTicket.TicketKey+" closed. "+
		"Be sure to update the ticket with basic information: "+foundTicket.URL+"\n"+
		"Use this postmortem template for detailed writeups: "+postmortemTemplate+"\n"+
		attachmentStatusMessage)
	return nil
}
