package incidents

/****************/ /* Channel Tracking Methods */ /****************/

import (
	// standard
	"io/ioutil"
	"log"
	"os"
	"path"
	"regexp"
	"strconv"
	"strings"
	"time"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// Check if a channel is being tracked and return it.
func (h *handler) isTrackedChannel(channelID string) (bool, *trackedChannel) {
	for _, trackedChannel := range h.trackedChannels {
		if trackedChannel.channelID == channelID {
			return true, trackedChannel
		}
	}
	return false, nil
}

// WARNING: This function will produce an index out of range error if h.cfg.ChatLogDir is a "bad folder", try /etc.
// Once fixed, remove this warning.
// NOTE: I think this function locks up. We have a debug log in the loop now. If it stops pritning every 10 min, investigate.
func (h *handler) checkTrackedChannels() error {
	// This function makes sure that the bot and JIRA and local tracking files are all in sync
	// If for some reason the bot is restarted we need to set up the state and file listeners / writers again
	// Similarly if the tracked incident is closed directly in JIRA we need to upload the log and stop tracking

	fileList, err := ioutil.ReadDir(h.cfg.ChatLogDir)
	if err != nil {
		return err
	}

	// Find all files that match the log file format and parse out the channelID and project key
	// This produces errors when passed a folder full of regular files.
	for _, fileInfo := range fileList {
		if matches := regexp.MustCompile(fileRegex).FindStringSubmatch(fileInfo.Name()); len(matches) == 3 && matches[1] != "" && matches[2] != "" {
			_, err := h.createTrackedChannel(matches[1], matches[2])
			if err != nil {
				log.Println("Error creating tracked channel", err)
			}
		}
	}

	// For all currently tracked channels, check JIRA to see if the Task has been closed directly,
	// if so, close the Incident here too.
	for _, tracked := range h.trackedChannels {
		go func(tc *trackedChannel) {
			ticket, err := h.jiraClient.GetTicket(tc.incidentKey)
			if err != nil {
				log.Print("Could not get ticket info")
				return
			}

			if ticket.Status != "Open" {
				// Pass the minimum amount of info we need, this will just send a
				// non-targeted message to the channel being tracked indicating closure
				if err := h.closeIncident(slack.Message{RespondTo: tc.channelID}, tc.incidentKey); err != nil {
					log.Printf("Could not autoclose issue [%v], %v", tc.incidentKey, err)
				} else {
					exp.Debug("Auto Closed Jira Incident: " + tc.incidentKey)
				}
			}
		}(tracked)
	}
	return nil
}

func (tc *trackedChannel) watchMessages(fh *os.File, slackClient slack.Client) {
	// This allows us to buffer messages if we need to do username / channel name lookups which might otherwise block
	tc.close = make(chan struct{})
	tc.messages = make(chan string, 10)
	// Watch these two channels in a go routine.
	go func() {
		defer func() {
			if err := fh.Close(); err != nil {
				log.Printf("Error closing file [%v] %v", fh.Name(), err)
			}
		}()

		for {
			select {
			case <-tc.close:
				return
			case logLine := <-tc.messages:
				// Users and channels will appear as slack identifiers, so for legibility
				// in the log we will replace them with human readable filenames
				words := strings.Split(logLine, " ")
				for i, word := range words {
					if matches := regexp.MustCompile(slack.UserRegex).FindStringSubmatch(word); len(matches) == 2 && matches[1] != "" {
						words[i] = "@" + slackClient.GetEntityName(matches[1])
					} else if matches := regexp.MustCompile(slack.ChannelRegex).FindStringSubmatch(word); len(matches) == 2 && matches[1] != "" {
						words[i] = "#" + slackClient.GetEntityName(matches[1])
					}
				}

				if _, err := fh.WriteString(strings.Join(words, " ") + "\n"); err != nil {
					// We're just going to log this and carry on,
					// TODO - work out how to measure / report on this.
					// Perhaps use expvar and a nagios check.
					log.Print("Could not write to log:", fh.Name())
				}
			}
		}
	}()
}

// Return a tracked channel from a Jira ticket ID.
func (h *handler) getTrackedChannelByJiraKey(ticketKey string) *trackedChannel {
	for _, tc := range h.trackedChannels {
		if tc.incidentKey == ticketKey {
			return tc
		}
	}
	return nil
}

func (tc *trackedChannel) logMessage(msg slack.Message) {
	// Format the log line with the timestamp and user that said the thing, slack user and channel IDs will be parsed out later
	i, err := strconv.ParseFloat(msg.Data.Timestamp, 64)
	if err != nil {
		exp.Debug("Error Parsing Timestamp: " + err.Error())
	}
	timestamp := time.Unix(int64(i), 0)
	if err != nil {
		exp.Debug("Error Parsing Timestamp: " + err.Error())
		timestamp = time.Now().UTC()
	}
	tc.messages <- timestamp.Format("2006-01-02T15:04:05Z") + " <@" + msg.User + "> " + msg.Data.Text
}

func (tc *trackedChannel) getLogPath(baseDir string) string {
	return path.Join(baseDir, tc.channelID+"-"+tc.incidentKey+".log")
}

func (h *handler) deleteTrackedChannel(ticketKey string) {
	var tcIndex int
	var foundChannel bool

	for i, trackedChannel := range h.trackedChannels {
		if trackedChannel.incidentKey == ticketKey {
			tcIndex = i
			close(trackedChannel.close)
			close(trackedChannel.messages)
			foundChannel = true
			break
		}
	}

	if !foundChannel {
		return
	}

	// Remove from list
	h.Lock()
	h.trackedChannels = append(h.trackedChannels[:tcIndex], h.trackedChannels[tcIndex+1:]...)
	h.Unlock()
}

func (h *handler) createTrackedChannel(channelID, incidentKey string) (*trackedChannel, error) {
	// This can happen when syncing state between the bot, files on disk, and JIRA. So just ignore if it's already set up
	for _, trackedChannel := range h.trackedChannels {
		if trackedChannel.channelID == channelID && trackedChannel.incidentKey == incidentKey {
			return trackedChannel, nil
		}
	}

	trackedChannel := &trackedChannel{channelID: channelID, incidentKey: incidentKey}
	h.Lock()
	h.trackedChannels = append(h.trackedChannels, trackedChannel)
	h.Unlock()
	fh, err := os.OpenFile(trackedChannel.getLogPath(h.cfg.ChatLogDir), os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		return trackedChannel, err
	}
	trackedChannel.watchMessages(fh, h.slackClient)
	return trackedChannel, nil
}
