package oncall

import (
	// standard
	"expvar"
	"sync"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

const (
	onCallCommand = "!oncall"
	serviceRegex  = `^(.*)\s*!oncall\s*(.*)$`
)

// Config Defines the data in the Glitch config for this handler.
type Config struct {
	APIToken         string          `toml:"pagerduty_api_token"` // Go get one from PD
	NobodyUserID     string          `toml:"nobody_user_id"`
	IgnoredPolicies  []string        `toml:"ignored_policies"`  // This needs to go to a database or something
	AdditionalPagers []onCallContact `toml:"additional_pagers"` // This needs to go to a database or something
}

// handler keeps track of everything we do. As a library, this can handle
// multiple pagerduty account integrations in one app.
type handler struct {
	cfg         *Config                      // glitch.conf settings
	slackClient slack.Client                 // Slack client interface
	pdClient    pagerduty.Client             // how we get data from PD
	services    []pagerduty.Service          // Cache
	policies    []pagerduty.EscalationPolicy // Cache
	oncalls     []pagerduty.OnCall           // Cache
	emails      map[string]string            // Cache
	export      exportData
	sync.Mutex
}

type exportData struct {
	expvar.Map                  // Top-key expvar export.
	calls         expvar.Int    // !oncall counter
	lastCall      exp.Time      // date of last !oncall
	lastDur       exp.Dur       // duration from !oncall to reply
	errors        expvar.Int    // error counter
	lastUser      expvar.String // last !oncall'er
	records       expvar.Int    // records sent to lastUser
	pdUpdate      exp.Time      // Last Cache Refresh ^
	pdEmailUpdate exp.Time      // Last EMail Refresh ^
}

// Is this a reply from pagerDuty? Is it really an "item" or should this be named onCallPolicy?
// TODO: Provide data example for each struct member.
type onCallResponseItem struct {
	EscalationPolicyName string
	Users                []onCallResponseUser
	ServiceName          string
	Description          string
	Email                string
	IntegrationID        string
	ServiceID            string
}

// Items have users? and/or schedules? This needs an explanation.
type onCallResponseUser struct {
	Name       string
	ID         string
	IsSchedule bool
}

// A contact is not a user? It's not clear why there is a contact and a user.
type onCallContact struct {
	Name        string `toml:"name"`
	User        string `toml:"user"`
	Email       string `toml:"email"`
	SearchTerms string `toml:"search_terms"`
}

// Used to satisfy the Sort() interface.
type onCallContacts []onCallContact

// we have "items", "users" "contacts" and now "integrations". mhm.
type integrationhelper struct {
	// capitalize these.
	integrationID      string
	serviceID          string
	serviceName        string
	serviceDescription string
}

// Where should these go? Give them proper comments so we know what they're used for.
// and create tests for them.
func (i *integrationhelper) ID() string {
	return i.serviceID + "|" + i.integrationID
}

func (i *onCallResponseItem) GetFQIntegrationID() string {
	return i.ServiceID + "|" + i.IntegrationID
}
