package oncall

import (
	// standard
	"sort"
	"strings"

	// external
	"github.com/crackcell/gotabulate"
)

/****************************************
       DATA FORMATTING FUNCTIONS
*****************************************/

// sortOnCalls turns data from pagerduty into our own, prioritized format.
func (h *handler) sortOnCalls(items []*onCallResponseItem) onCallContacts {
	c := make(onCallContacts, 0)

	for _, item := range items {
		listItem := onCallContact{
			Name:  item.EscalationPolicyName,
			Email: item.Email,
		}
		if item.ServiceName != "" {
			listItem.Name = item.ServiceName
			// we use the service description from pagerduty to specify additional search terms
			listItem.SearchTerms = item.Description
		}

		scheduleEntry := h.getScheduleEntry(item.Users)
		if listItem.User = item.Users[0].Name; scheduleEntry != nil {
			listItem.User = scheduleEntry.Name
		}

		c = append(c, listItem)
	}

	sort.Sort(c)
	return c
}

// Len Satisifies the Sort interface.
func (c onCallContacts) Len() int {
	return len(c)
}

// Less Satisifies the Sort interface.
func (c onCallContacts) Less(i, j int) bool {
	return strings.ToLower(c[i].Name) < strings.ToLower(c[j].Name)
}

// Swap Satisifies the Sort interface.
func (c onCallContacts) Swap(i, j int) {
	c[i], c[j] = c[j], c[i]
}

// formatOnCalls uses tabulator to make pretty messages from large chunks of data.
func formatOnCalls(items onCallContacts) []string {
	itemsForTabulate := make([][]string, 0)
	// We can only fit a certain number of lines per message
	// Tabulate first then split by page so that each message is formatted with the same spacing
	for _, item := range items {
		row := []string{item.Name, item.User, item.Email}
		itemsForTabulate = append(itemsForTabulate, row)
	}
	tabulator := gotabulate.NewTabulator()
	tabulator.SetFirstRowHeader(true)
	tabulator.SetFormat("plain")
	lines := strings.Split(tabulator.Tabulate(itemsForTabulate), "\n")

	msg := "Currently on call:\n```\n"
	msgs := make([]string, 0)
	for _, line := range lines {
		if line == "" {
			continue
		}

		// If appending this line (as well as all the extra formatting that might get appended later)
		// would result in a message longer than the max, then end this chunk here and build a new message
		if len(line+"\n```")+len(msg) > 4000 {
			// close this message
			msgs = append(msgs, msg+"```")
			// Reset the next message-set
			msg = "(continued...)\n```\n" + line + "\n"
		} else {
			msg += line + "\n"
		}
	}
	msgs = append(msgs, msg+"```")

	return msgs
}
