package oncall


import (
	// standard
	"errors"
	"expvar"
	"log"
	"strings"
	"time"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// Init is called during setup of all handlers and is used to register the correct callbacks by returning a common.Command
func Init(config *Config, slackClient slack.Client, pdClient pagerduty.Client) slack.Handler {
	handler := &handler{
		cfg:         config,
		slackClient: slackClient,
		pdClient:    pdClient,
	}
	handler.export.Init()
	handler.export.Set("!oncall_counter", &handler.export.calls)
	handler.export.Set("!oncall_last_call", &handler.export.lastCall)
	handler.export.Set("!oncall_errors", &handler.export.errors)
	handler.export.Set("!oncall_last_caller", &handler.export.lastUser)
	handler.export.Set("oncall_records", &handler.export.records)
	handler.export.Set("last_query_duration", &handler.export.lastDur)
	handler.export.Set("last_pd_update", &handler.export.pdUpdate)
	log.Print("OnCall Handler Loaded! Commands: !oncall")
	return handler
}

// IsMatch tests the slack message for our command(s).
func (h *handler) IsMatch(msg slack.Message) bool {
	return msg.Data.Text == onCallCommand || strings.HasPrefix(msg.Data.Text, onCallCommand+" ")
}

// Export returns the expvar map for our debug and health information.
func (h *handler) Export() *expvar.Map {
	return &h.export.Map
}

func (h *handler) Name() string {
	// TODO: turn this back on when we properly parse Pagerduty for vidcs
	return "oncall"
}

// Execute runs when someone sends !oncall
func (h *handler) Execute(msg slack.Message) error {
	log.Print("Processing !oncall request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", h.slackClient.GetEntityName(msg.RespondTo))
	h.slackClient.SendReply(msg, "Preparing On Call List! Response will be sent in DM.")
	// This lock means the bot can only respond to one !oncall request at a time.
	// That's okay because it caches the pagerduty responses for a bit.
	// The lock also protects race conditions in our maps.
	h.Lock()
	defer h.Unlock()
	h.export.calls.Add(1)
	h.export.lastCall.Now()
	h.export.lastUser.Set(h.slackClient.GetEntityName(msg.Data.User))
	h.refreshPDConfigs()

	if len(h.oncalls) < 1 {
		h.export.errors.Add(1)
		return errors.New("error retrieving oncalls")
	} else if len(h.policies) < 1 {
		h.export.errors.Add(1)
		return errors.New("error retrieving policies")
	} else if len(h.services) < 1 {
		h.export.errors.Add(1)
		return errors.New("error retrieving services")
	}

	/* Gather PagerDuty Data */
	items := h.getOnCallItems()
	integrationIDs := h.getIntegIDs(items)
	h.populateIntegrationEmails(items, integrationIDs)
	contacts := h.sortOnCalls(items)

	// There are some config entries that don't come from pagerduty,
	// we should probably find a better way of supporting these, like a db
	contacts = append(contacts, h.cfg.AdditionalPagers...)

	reply := h.buildReply(contacts, msg.Data.Text)

	/* Send off the reponse. */
	// Grab the DM channel ID for the user.
	replyID, err := h.slackClient.GetUserDMChannelID(msg.Data.User)
	if err != nil { // this is never true.
		replyID = msg.RespondTo
	}
	for _, r := range reply {
		h.slackClient.SendMessage(replyID, []string{}, r)
		time.Sleep(100 * time.Millisecond)
	}
	return nil
}

// Stop is called when the app reloads.
func (h *handler) Stop() error {
	// Nothing to stop.
	return nil
}

// StringInList returns true if the string is in the slice of strings.
func StringInList(FindMe string, TheList []string) bool {
	for _, l := range TheList {
		if FindMe == l {
			return true
		}
	}
	return false
}
