package oncall

import (
	// standard
	"errors"
	"testing"

	// external
	"github.com/stretchr/testify/assert"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty/pagerdutyfakes"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack/slackfakes"
	//"github.com/PagerDuty/go-pagerduty"
)

func generateObjectReference(id, summary, objType string) pagerduty.ObjectReference {
	return pagerduty.ObjectReference{ID: id, Summary: summary, Type: objType}
}

var testOnCallList = []pagerduty.OnCall{
	pagerduty.OnCall{
		EscalationLevel:  1,
		EscalationPolicy: generateObjectReference("EP1ID", "EP1SUMMARY FindMe", ""),
		UserSummary:      "USER1SUMMARY",
		ScheduleID:       "SCHEDULE1ID",
	},
	pagerduty.OnCall{
		EscalationLevel:  1,
		EscalationPolicy: generateObjectReference("EP2ID", "EP2SUMMARY", ""),
		UserSummary:      "USER2SUMMARY",
		UserID:           "real_user_id",
		ScheduleID:       "SCHEDULE2ID",
	},
	pagerduty.OnCall{
		EscalationLevel:  1,
		EscalationPolicy: generateObjectReference("EP2ID", "EP2SUMMARY", ""),
		UserSummary:      "NOBODY",
		UserID:           "nobody_user_id",
		ScheduleID:       "SCHEDULENOBODYID",
	},
	pagerduty.OnCall{
		EscalationLevel:  1,
		EscalationPolicy: generateObjectReference("EP3ID", "EP3SUMMARY", ""),
		UserSummary:      "USER3SUMMARY",
		ScheduleID:       "",
	},
	pagerduty.OnCall{
		EscalationLevel:  1,
		EscalationPolicy: generateObjectReference("EPIGNOREID", "EPIGNORESUMMARY", ""),
		UserSummary:      "USERIGNOREDSUMMARY",
		ScheduleID:       "SCHEDULEIGNOREDID",
	},
}

var testEscalationPoliciesList = []pagerduty.EscalationPolicy{
	pagerduty.EscalationPolicy{
		ID:      "EP1ID",
		Summary: "EP1SUMMARY FindMe",
		Services: []pagerduty.ObjectReference{
			generateObjectReference("SVC1ID", "", "SVC1TYPE"),
			generateObjectReference("SVC2ID", "", "SVC2TYPE"),
			generateObjectReference("SVCNOTFOUNDID", "", "SVCNOTFOUNDTYPE"),
		},
	},
	pagerduty.EscalationPolicy{
		ID:      "EP2ID",
		Summary: "EP2SUMMARY",
		Services: []pagerduty.ObjectReference{
			generateObjectReference("SVC3ID", "", "SVC3TYPE"),
			generateObjectReference("SVCIGNOREDID", "", "SVCIGNOREDTYPE"),
		},
	},
	pagerduty.EscalationPolicy{
		ID:      "EP3ID",
		Summary: "EP3SUMMARY",
		Services: []pagerduty.ObjectReference{
			generateObjectReference("SVC2ID", "", "SVC2TYPE"),
			generateObjectReference("SVC3ID", "", "SVC3TYPE"),
		},
	},
}

var testServicesList = []pagerduty.Service{
	pagerduty.Service{
		ID:          "SVC1ID",
		Name:        "SVC1NAME",
		Description: "SVC1DESCRIPTION",
		Integrations: []pagerduty.ObjectReference{
			generateObjectReference("INTEG1ID", "", "generic_email_inbound_integration_reference"),
			generateObjectReference("INTEG2ID", "", "generic_email_inbound_integration_reference"),
		},
	},
	pagerduty.Service{
		ID:          "SVC2ID",
		Name:        "SVC2NAME",
		Description: "SVC2DESCRIPTION",
		Integrations: []pagerduty.ObjectReference{
			generateObjectReference("INTEG3ID", "", "generic_email_inbound_integration_reference"),
			generateObjectReference("INTEG4ID", "", "non_email_type"),
		},
	},
	pagerduty.Service{
		ID:          "SVC3ID",
		Name:        "SVC3NAME FindMe !oncall",
		Description: "SVC3DESCRIPTION",
		Integrations: []pagerduty.ObjectReference{
			generateObjectReference("INTEG5ID", "", "generic_email_inbound_integration_reference"),
		},
	},
}

var testIntegraionList = map[string]string{
	"INTEG1ID": "integ1@email.com",
	"INTEG2ID": "integ2@email.com",
	"INTEG3ID": "integ3@email.com",
	"INTEG4ID": "integ4@email.com",
	"INTEG5ID": "integ5@email.com",
}

var testResponse1 = "Currently on call:\n```\nEP1SUMMARY FindMe   USER1SUMMARY   integ1@email.com   \nEP2SUMMARY          USER2SUMMARY   integ5@email.com   \nSVC3NAME FindMe     USER3SUMMARY   integ5@email.com   \nADD1                    ADD1USER          ADD1EMAIL   \n```"

var testResponse2 = "Currently on call:\n```\nEP1SUMMARY FindMe   USER1SUMMARY   integ1@email.com   \nSVC3NAME FindMe     USER3SUMMARY   integ5@email.com   \n```"

// This only does a partial cover because we have no slack connection in tests.
func TestIsMatch(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	// Mock data
	config := &Config{APIToken: "none", NobodyUserID: "nobody_user_id"}
	slackClient := new(slackfakes.FakeClient)
	pdClient := new(pagerdutyfakes.FakeClient)

	handler := Init(config, slackClient, pdClient)
	msg := slack.Message{}

	// Do the test!
	msg.Data.Text = "!oncall"
	a.True(handler.IsMatch(msg), "handler must return true")
	msg.Data.Text = "!notforus"
	a.False(handler.IsMatch(msg), "handler must return false")
}

func TestOnCallResponse(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// Mock data
	config := &Config{APIToken: "none", NobodyUserID: "nobody_user_id", IgnoredPolicies: []string{"EPIGNORESUMMARY"}, AdditionalPagers: []onCallContact{
		onCallContact{
			Name:  "ADD1",
			User:  "ADD1USER",
			Email: "ADD1EMAIL",
		}}}
	slackClient := new(slackfakes.FakeClient)
	pdClient := new(pagerdutyfakes.FakeClient)

	pdClient.GetIntegrationStub = func(serviceID string, integrationID string) (pagerduty.Integration, error) {
		// Just testing so use the integration ID as the sole key
		if email, ok := testIntegraionList[integrationID]; ok {
			return pagerduty.Integration{
				ID:               serviceID + "|" + integrationID,
				IntegrationEmail: email,
			}, nil
		}

		return pagerduty.Integration{}, nil
	}

	msg := slack.Message{}
	msg.Data.Text = "!oncall"
	msg.Data.User = "USERID1"
	msg.RespondTo = "CHANNELID1"

	handler := Init(config, slackClient, pdClient)
	a.EqualValues("oncall", handler.Name())

	exportMap := handler.Export()

	// Test empty oncalls
	err := handler.Execute(msg)

	_, reply := slackClient.SendReplyArgsForCall(0)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.EqualValues(err.Error(), "error retrieving oncalls")
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("1", exportMap.Get("!oncall_errors").String())

	// We reinit the handler to reset cache timers
	handler = Init(config, slackClient, pdClient)
	exportMap = handler.Export()

	pdClient.ListOnCallsReturns(testOnCallList, nil)

	// Test empty policies
	err = handler.Execute(msg)

	_, reply = slackClient.SendReplyArgsForCall(1)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.EqualValues(err.Error(), "error retrieving policies")
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("1", exportMap.Get("!oncall_errors").String())

	// We reinit the handler to reset cache timers
	handler = Init(config, slackClient, pdClient)
	exportMap = handler.Export()

	pdClient.ListEscalationPoliciesReturns(testEscalationPoliciesList, nil)

	// Test empty services
	err = handler.Execute(msg)

	_, reply = slackClient.SendReplyArgsForCall(2)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.EqualValues(err.Error(), "error retrieving services")
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("1", exportMap.Get("!oncall_errors").String())

	// We reinit the handler to reset cache timers
	handler = Init(config, slackClient, pdClient)
	exportMap = handler.Export()

	// Test normal call with invalid DM user
	pdClient.ListServicesReturns(testServicesList, nil)
	slackClient.GetUserDMChannelIDReturns("", errors.New("Couldn't get DM channel ID"))
	err = handler.Execute(msg)

	_, reply = slackClient.SendReplyArgsForCall(3)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.Nil(nil)
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("0", exportMap.Get("!oncall_errors").String())

	respondTo, recipient, message := slackClient.SendMessageArgsForCall(0)
	a.EqualValues("CHANNELID1", respondTo)
	a.EqualValues([]string{}, recipient)
	a.EqualValues(testResponse1, message)

	// Test with correct DM Channel ID
	slackClient.GetUserDMChannelIDReturns("DMCHANNELID1", nil)
	err = handler.Execute(msg)
	respondTo, _, message = slackClient.SendMessageArgsForCall(1)
	a.EqualValues("DMCHANNELID1", respondTo)
	a.EqualValues(testResponse1, message)

	// Test with search term
	msg.Data.Text = "!oncall FindMe"
	handler = Init(config, slackClient, pdClient)
	exportMap = handler.Export()

	err = handler.Execute(msg)

	_, reply = slackClient.SendReplyArgsForCall(4)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.Nil(err)
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("0", exportMap.Get("!oncall_errors").String())

	respondTo, _, message = slackClient.SendMessageArgsForCall(2)
	a.EqualValues("DMCHANNELID1", respondTo)
	a.EqualValues(testResponse2, message)

	// Test with bogus search term
	msg.Data.Text = "!oncall Don'tFindMe"
	handler = Init(config, slackClient, pdClient)
	exportMap = handler.Export()

	err = handler.Execute(msg)

	_, reply = slackClient.SendReplyArgsForCall(5)
	a.EqualValues("Preparing On Call List! Response will be sent in DM.", reply)
	a.Nil(err)
	a.EqualValues("1", exportMap.Get("!oncall_counter").String())
	a.EqualValues("0", exportMap.Get("!oncall_errors").String())

	respondTo, _, message = slackClient.SendMessageArgsForCall(3)
	a.EqualValues("DMCHANNELID1", respondTo)
	a.EqualValues("Unable to locate oncall entries matching 'Don'tFindMe'. The full list follows.", message)

	respondTo, _, message = slackClient.SendMessageArgsForCall(4)
	a.EqualValues("DMCHANNELID1", respondTo)
	a.EqualValues(testResponse1, message)

	// Test stop for completeness
	a.EqualValues(nil, handler.Stop())
}
