package oncall

/* PagerDuty specific functions for Glitch !oncall handler. */

import (
	// standard
	"log"
	"regexp"
	"strings"
	"time"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty"
)

// TODO: break this down more.
func (h *handler) getOnCallItems() []*onCallResponseItem {
	items := make([]*onCallResponseItem, 0)
	for _, oc := range h.oncalls {
		// Only return primary oncalls for a given escalation policy, also ignore policies specified to be ignored in the config
		if oc.EscalationLevel != 1 || StringInList(oc.EscalationPolicy.Summary, h.cfg.IgnoredPolicies) {
			continue
		}

		// It's possible for multiple entries to exist for the same Escalation Policy if multiple entities are on call at the same time
		// So we'll match on escalation policy name and create a list of users, we pick the "correct" one later on depending on what's available
		item := getOrCreateItem(&items, oc.EscalationPolicy.Summary)
		user := &onCallResponseUser{
			Name:       oc.UserSummary,
			ID:         oc.UserID,
			IsSchedule: oc.ScheduleID != "",
		}

		item.Users = append(item.Users, *user)

		p := getPolicy(oc.EscalationPolicy.ID, h.policies)

		// Find all the services that have an email itegration
		/*
			This code is really hacky, but we're replacing the old glitch bot with this new version which uses a different version of the PD API
			As such we have to do some weird stuff to make it work the same way as the old one because the new API doesn't support the method that the
			old API did, we will eventually change the way that !oncall emails are designated by tagging them as such directly in PagerDuty
			but until that time, we need to do this weird code to emulate the old functionality and this code will also work with the new way of
			designating services for use in the !oncall command, once we've tidied up all teams' pagerduty setups to use the new way of designating services
			we can clean up this code massively
		*/

		emailServiceHelpers := make([]*integrationhelper, 0)

		// Loop Services for each Oncall.
		for _, svc := range p.Services {
			service := getService(svc.ID, h.services)
			if service == nil {
				log.Println("Error looking up service from ID", svc.ID)
				continue
			}

			// Loop Integrations for each service.
			for _, integration := range service.Integrations {
				if integration.Type == "generic_email_inbound_integration_reference" {
					// We need to make individual calls to the API :( to get each integration so we're just storing IDs here
					// So we can get them in parallel later
					emailServiceHelpers = append(emailServiceHelpers, &integrationhelper{
						integrationID:      integration.ID,
						serviceID:          service.ID,
						serviceName:        service.Name,
						serviceDescription: service.Description,
					})
					break
				}
			}
		}

		var oncallServiceIntegration *integrationhelper

		if len(emailServiceHelpers) == 1 {
			// If we've only got one service with an email address then use it
			oncallServiceIntegration = emailServiceHelpers[0]
		} else if len(emailServiceHelpers) > 1 {
			for _, integrationhelper := range emailServiceHelpers {
				// If the service has been explicitly designated for use in the
				// !oncall command then use it and break
				if regexp.MustCompile(serviceRegex).MatchString(integrationhelper.serviceName) {
					oncallServiceIntegration = integrationhelper
					// This is up here because we only want the ServiceName to be set in this scenario, not the one below
					item.ServiceName = strings.Trim(strings.Replace(oncallServiceIntegration.serviceName, "!oncall", "", -1), " ")
					break
				}

				// We're going to use the length of the service name as a proxy for the service who's email we should use in the !oncall listing
				// It's a bit hacky but it works when comparing it to the output of the old glitch bot
				if oncallServiceIntegration == nil || len(integrationhelper.serviceName) < len(oncallServiceIntegration.serviceName) {
					oncallServiceIntegration = integrationhelper
				}
			}
		}

		// Now that we know which service we're supposed to use, set the values
		// we've worked out above on the item so that we can do the lookup we need to do
		if oncallServiceIntegration != nil {
			item.ServiceID = oncallServiceIntegration.serviceID
			item.IntegrationID = oncallServiceIntegration.integrationID
			item.Description = oncallServiceIntegration.serviceDescription
		}
	} // Closes the oncall loop.
	return items
}

// TODO: what's this do?
func (h *handler) getIntegIDs(items []*onCallResponseItem) (integrationIDs []integrationhelper) {
	for _, item := range items {
		if item.IntegrationID != "" {
			integrationIDs = append(integrationIDs, integrationhelper{
				integrationID: item.IntegrationID,
				serviceID:     item.ServiceID,
			})
		}
	}
	return
}

// filterContacts returns a slice of contacts that match a filter.
func filterContacts(contacts onCallContacts, filter string) (onCallContacts, bool, string) {
	filteredContacts := make([]onCallContact, 0)
	for _, item := range contacts {
		// We want to search the name and additional search terms for matches
		if strings.Contains(strings.ToLower(item.Name+" "+item.SearchTerms), strings.ToLower(filter)) {
			filteredContacts = append(filteredContacts, item)
		}
	}
	if len(filteredContacts) > 0 {
		return filteredContacts, false, filter
	}
	return contacts, true, filter
}

// buildReply takes all the contacts, filters them if requested and returns a
// slice of strings to send as replies.
func (h *handler) buildReply(contacts onCallContacts, filter string) []string {
	filterFailed := false
	if parts := strings.SplitN(strings.Trim(filter, " \n"), " ", 2); len(parts) == 2 && parts[1] != "" {
		// If we've been passed a search term, use it.
		contacts, filterFailed, filter = filterContacts(contacts, parts[1])
	}

	// Update exported data.
	h.export.records.Set(int64(len(contacts)))
	h.export.lastDur.Set(time.Now().Sub(h.export.lastCall.Value()))
	formattedContacts := formatOnCalls(contacts)
	if filterFailed {
		// Prepend an "error" message to formattedContacts because the filter failed.
		formattedContacts = append(formattedContacts, "")
		copy(formattedContacts[1:], formattedContacts)
		formattedContacts[0] = "Unable to locate oncall entries matching '" + filter + "'. The full list follows."
	}
	return formattedContacts
}

func getPolicy(id string, policies []pagerduty.EscalationPolicy) *pagerduty.EscalationPolicy {
	for _, policy := range policies {
		if policy.ID == id {
			return &policy
		}
	}
	return nil
}

func getService(id string, services []pagerduty.Service) *pagerduty.Service {
	for _, service := range services {
		if service.ID == id {
			return &service
		}
	}
	return nil
}

func (h *handler) refreshPDConfigs() {
	if h.export.pdUpdate.Value().After(time.Now().Add(time.Minute * -4)) {
		// Only update every X (4) minutes.
		return
	}
	var err error

	h.oncalls, err = h.pdClient.ListOnCalls()
	if err != nil {
		log.Println("Error listing OnCalls", err)
		return
	}

	h.policies, err = h.pdClient.ListEscalationPolicies()
	if err != nil {
		log.Println("Error listing Escalation Policies", err)
		return
	}

	h.services, err = h.pdClient.ListServices()
	if err != nil {
		log.Println("Error listing Services", err)
		return
	}

	h.export.pdUpdate.Now()
}

func getOrCreateItem(items *[]*onCallResponseItem, escalationPolicyName string) *onCallResponseItem {
	// We can have multiple entries of the same escalation policy if multiple entities are on call
	// for the same escalation policy at the same time
	for _, item := range *items {
		if item.EscalationPolicyName == escalationPolicyName {
			return item
		}
	}

	// Create a new onCallResponseItem with the supplied escalation policy name
	item := &onCallResponseItem{
		EscalationPolicyName: escalationPolicyName,
		Users:                make([]onCallResponseUser, 0),
	}

	*items = append(*items, item)

	return item
}

func (h *handler) getScheduleEntry(u []onCallResponseUser) *onCallResponseUser {
	var retUser onCallResponseUser
	for _, user := range u {
		// If there's multiple schedule entries try to get one that has a real person associated with it
		if user.IsSchedule && (retUser.ID == "" || retUser.ID == h.cfg.NobodyUserID) {
			retUser = user
		}
	}

	if retUser.ID == "" {
		return nil
	}
	return &retUser
}

// Lets find a way to run this once at startup, and lets find a better thing to call an oncallresponseitem.
func (h *handler) populateIntegrationEmails(items []*onCallResponseItem, ids []integrationhelper) {
	// We don't expect these to be updated very often at all so cache for a while, this really improves performance
	// because we have to set the email address for every item in the list which involves individual calls to the PD API
	// becase it doesn't currently have a way to get these in bulk
	if h.export.pdEmailUpdate.Value().Before(time.Now().Add(time.Hour * -6)) {
		// Only update every X (6) hours.
		h.emails = make(map[string]string)
		for _, integ := range ids {
			h.updatePdIntegrationEmail(integ)
		}
		h.export.pdEmailUpdate.Now()
	}

	for _, item := range items {
		// It's possible a new email was added during the cached period so handle that case here
		if email, ok := h.emails[item.GetFQIntegrationID()]; ok {
			item.Email = email
		} else if item.IntegrationID != "" {
			item.Email = h.updatePdIntegrationEmail(integrationhelper{serviceID: item.ServiceID, integrationID: item.IntegrationID})
		} else {
			item.Email = "No pager email specified"
		}
	}
}

func (h *handler) updatePdIntegrationEmail(integ integrationhelper) string {
	var integrationEmail string

	integration, err := h.pdClient.GetIntegration(integ.serviceID, integ.integrationID)

	if err != nil {
		// Two different error messages seems confusing, but one is a debugging one and the other is a more basic one for users
		log.Println("Error retrieving integration", integ.ID())
		integrationEmail = "Error retrieving email address"
	} else {
		integrationEmail = integration.IntegrationEmail
	}

	h.emails[integ.ID()] = integrationEmail

	return integrationEmail
}
