package tickets

import (
	// standard
	"expvar"
	"fmt"
	"log"
	"regexp"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// Init is called during setup of all commands and is used to register the correct callbacks by returning a common.Command
func Init(config *Config, slackClient slack.Client, jiraClient jira.Client) slack.Handler {
	handler := &handler{
		cfg:         config,
		slackClient: slackClient,
		jiraClient:  jiraClient,
	}
	handler.export.Init()
	log.Println("Tickets Handler Loaded!", len(config.Channels), "channels configured. Commands: !ticket")
	return handler
}

func (h *handler) Export() *expvar.Map {
	return &h.export.Map
}

func (h *handler) IsMatch(msg slack.Message) bool {
	return regexp.MustCompile(commandRegex).MatchString(msg.Data.Text)
}

func (h *handler) Name() string {
	return "tickets"
}

func (h *handler) Execute(msg slack.Message) error {
	entityName := h.slackClient.GetEntityName(msg.RespondTo)

	log.Print("Processing '!ticket' request from @", h.slackClient.GetEntityName(msg.Data.User), " in #", entityName)

	if channelConfigs, ok := h.cfg.Channels[entityName]; ok {
		matches := regexp.MustCompile(commandRegex).FindStringSubmatch(msg.Data.Text)
		if len(matches) < 2 || matches[1] == "" {
			h.slackClient.SendReply(msg, "Please supply a ticket description")
			return nil
		}

		var projectKey string
		var description string
		var chanConfig channelConfig

		if len(channelConfigs) > 1 {
			// Comment some of this code.
			projectMatches := regexp.MustCompile(keySuppliedRegex).FindStringSubmatch(matches[1])
			if len(projectMatches) < 3 || projectMatches[1] == "" || projectMatches[2] == "" {
				h.slackClient.SendReply(msg, "Please supply a project key and ticket description")
				return nil
			}

			projectKey = projectMatches[1]
			description = projectMatches[2]

			for _, cc := range channelConfigs {
				if projectKey == cc.ProjectKey {
					chanConfig = cc
					break
				}
			}

			if chanConfig == (channelConfig{}) {
				h.slackClient.SendReply(msg, fmt.Sprintf("This channel is not set up to handle !tickets for the supplied project [%v]", projectKey))
				return nil
			}
		} else {
			chanConfig = channelConfigs[0]
			description = matches[1]
		}

		h.slackClient.SendReply(msg, "Creating ticket")

		ticket := &jira.Ticket{
			Summary:    description,
			IssueType:  chanConfig.IssueType,
			ProjectKey: chanConfig.ProjectKey,
		}
		err := h.jiraClient.CreateTicket(ticket)

		if err != nil {
			return err
		}

		h.slackClient.SendReply(msg, fmt.Sprintf("Opened new ticket: %v: %v", ticket.TicketKey, ticket.URL))
		return nil
	}

	h.slackClient.SendReply(msg, "This channel is not set up to handle !tickets, the Systems team can help you get this set up.")

	return nil
}

// Stop is called when the app reloads.
func (h *handler) Stop() error {
	// Nothing to stop.
	return nil
}
