package tickets

import (
	// standard
	"fmt"
	"testing"

	// external
	"github.com/stretchr/testify/assert"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira/jirafakes"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack/slackfakes"
)

/*
NOTE: I wrote these tests quickly. They do not test the full gamut of what this
handler should do. We need to make a few helper functions to make writing the
tests easier. Then we can check more data from the fake functions being called.
*/

// This only does a partial cover because we have no slack connection in tests.
func TestIsMatch(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// Mock the data
	channels := map[string][]channelConfig{
		"XOXO": []channelConfig{{ProjectKey: "TEST", IssueType: "Task"}},
	}
	config := &Config{Channels: channels}
	handler := Init(config, nil, nil)
	msg := slack.Message{}

	// Do the test!
	msg.Data.Text = "!ticket"
	a.True(handler.IsMatch(msg), "handler must return true")
	msg.Data.Text = "!notforus"
	a.False(handler.IsMatch(msg), "handler must return false")
}

// Make sure Stop() doesn't produce vomit, somehow.
func TestStop(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// Mock the data
	channels := map[string][]channelConfig{
		"XOXO": []channelConfig{{ProjectKey: "TEST", IssueType: "Task"}},
	}
	config := &Config{Channels: channels}
	handler := Init(config, nil, nil)
	a.Nil(handler.Stop(), "Stop must not return an error")
}

func TestExecute(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// Mock the data
	channels := map[string][]channelConfig{
		"XOXO":           []channelConfig{{ProjectKey: "TEST", IssueType: "Task"}},
		"PRIVATECHANNEL": []channelConfig{{ProjectKey: "TEST", IssueType: "Task"}},
	}
	config := &Config{Channels: channels}
	// With these, we control ALL the ouput from our slack and jira methods.
	jiraClient := new(jirafakes.FakeClient)
	slackClient := new(slackfakes.FakeClient)
	handler := Init(config, slackClient, jiraClient)
	msg := slack.Message{
		RespondTo: "C12345678",
	}

	// Check default call with no config
	slackClient.GetEntityNameReturns("NOTXOXO")
	msg.Data.Text = "!ticket"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(1, slackClient.SendReplyCallCount())
	_, sent := slackClient.SendReplyArgsForCall(0)
	a.EqualValues("This channel is not set up to handle !tickets, the Systems team can help you get this set up.", sent)

	// Test with no parameters
	slackClient.GetEntityNameReturns("XOXO")
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(2, slackClient.SendReplyCallCount())
	_, sent = slackClient.SendReplyArgsForCall(1)
	a.EqualValues("Please supply a ticket description", sent)

	// Test private channel with no parameters
	slackClient.GetEntityNameReturns("PRIVATECHANNEL")
	msg.RespondTo = "G12345678"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(3, slackClient.SendReplyCallCount())
	_, sent = slackClient.SendReplyArgsForCall(2)
	a.EqualValues("Please supply a ticket description", sent)

	// Make a real Ticket.
	slackClient.GetEntityNameReturns("XOXO")
	msg.Data.Text = "!ticket GO DADDY SUPPORT"
	msg.RespondTo = "C12345678"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(5, slackClient.SendReplyCallCount())
	_, sent = slackClient.SendReplyArgsForCall(3)
	a.EqualValues("Creating ticket", sent)
	_, sent = slackClient.SendReplyArgsForCall(4)
	a.EqualValues("Opened new ticket: : ", sent)

	// Test ticket creation failure
	jiraClient.CreateTicketReturns(fmt.Errorf("FAKE TICKET FAILED"))
	a.EqualError(handler.Execute(msg), "FAKE TICKET FAILED")

	// Make sure Export returns something
	a.NotNil(handler.Export())
}

// Parallel for speed.
func TestExecute2(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// Try multiple channels.
	channels := map[string][]channelConfig{
		"XOXO": []channelConfig{
			{ProjectKey: "INC", IssueType: "Incident"},
			{ProjectKey: "VT", IssueType: "Task"},
		},
	}
	config := &Config{Channels: channels}
	jiraClient := new(jirafakes.FakeClient)
	slackClient := new(slackfakes.FakeClient)
	handler := Init(config, slackClient, jiraClient)
	msg := slack.Message{
		RespondTo: "C12345678",
	}
	slackClient.GetEntityNameReturns("XOXO")

	msg.Data.Text = "!ticket GO DADDY SUPPORT"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(1, slackClient.SendReplyCallCount())
	_, sent := slackClient.SendReplyArgsForCall(0)
	a.EqualValues("This channel is not set up to handle !tickets for the supplied project [GO]", sent)
	config.Channels = map[string][]channelConfig{
		"XOXO": []channelConfig{
			{ProjectKey: "INC", IssueType: "Incident"},
			{ProjectKey: "VT", IssueType: "Task"},
		},
	}
	msg.Data.Text = "!ticket INC GO DADDY SUPPORT"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(3, slackClient.SendReplyCallCount())
	_, sent = slackClient.SendReplyArgsForCall(1)
	a.EqualValues("Creating ticket", sent)
	_, sent = slackClient.SendReplyArgsForCall(2)
	a.EqualValues("Opened new ticket: : ", sent)

	msg.Data.Text = "!ticket INC"
	a.Nil(handler.Execute(msg), "Execute must not return an error")
	a.EqualValues(4, slackClient.SendReplyCallCount())
	_, sent = slackClient.SendReplyArgsForCall(3)
	a.EqualValues("Please supply a project key and ticket description", sent)
}

func TestName(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	config := &Config{}
	jiraClient := new(jirafakes.FakeClient)
	slackClient := new(slackfakes.FakeClient)
	handler := Init(config, slackClient, jiraClient)
	a.EqualValues("tickets", handler.Name(), "package name is wrong?")
}
