package exp

// Helper functions for expvar.

import (
	// standard
	"encoding/json"
	"expvar"
	"fmt"
	"log"
	"reflect"
	"sync"
	"time"

	// external
	"github.com/pkg/errors"
)

// DebugMap identifies the name of the DebugLog map.
const DebugMap = "exp.Debug"

// We hold a list of map pointers here, so we can retain data through reload.
var maps Maps

// DebugLog turns on logging (syslog) for debug messages.
var DebugLog bool

// Maps holds a list of reusable expvar maps.
type Maps struct {
	list map[string]*expvar.Map
	sync.Mutex
}

/***********************************************/
//              exp helper funcs               //
/***********************************************/

// Debug is used to print and store debug messages. Call Init() on ExportMap first.
func Debug(msg string) string {
	DebugMap := GetMap(DebugMap)
	var debugMsg expvar.String
	debugMsg.Set(msg)
	// Maybe add a better time Format().
	timeKey := time.Now().String()
	DebugMap.Set(timeKey, &debugMsg)
	if DebugLog {
		log.Println("DEBUG:", msg)
	}
	return timeKey
}

// GetMap returns an unpublished map if one exists, or returns a new one.
func GetMap(name string) *expvar.Map {
	maps.Lock()
	defer maps.Unlock()
	if maps.list == nil {
		maps.list = make(map[string]*expvar.Map)
	}
	if m, mapExists := maps.list[name]; mapExists {
		return m
	}
	maps.list[name] = new(expvar.Map).Init()
	return maps.list[name]
}

// GetPublishedMap returns a published map if one exists, or returns a new one.
func GetPublishedMap(name string) *expvar.Map {
	if p := expvar.Get(name); p != nil {
		return p.(*expvar.Map)
	}
	p := GetMap(name)
	expvar.Publish(name, p)
	return p
}

// GetMapList returns a list of all the map names.
func GetMapList() (maplist []string) {
	maps.Lock()
	defer maps.Unlock()
	if maps.list == nil {
		return
	}
	for m := range maps.list {
		maplist = append(maplist, m)
	}
	return
}

// GetMapValuesString Turns an expvar map into a map of string/strings.
// Useful when formatting the data for slack.
func GetMapValuesString(name string) (map[string]string, error) {
	var raw map[string]interface{}
	if err := json.Unmarshal([]byte(GetMap(name).String()), &raw); err != nil {
		return nil, errors.Wrap(err, "json.Unmarshal")
	}
	list := make(map[string]string)
	for k, v := range raw {
		// Ignore nested maps. They should be generated independently.
		if reflect.ValueOf(v).Kind() != reflect.Map {
			list[k] = fmt.Sprint(v)
		}
	}
	return list, nil
}

// MakePrettyJSONString adds tabs to a raw json string.
func MakePrettyJSONString(inputJSON string) ([]byte, error) {
	var raw map[string]interface{}
	data := []byte(inputJSON)
	if err := json.Unmarshal(data, &raw); err != nil {
		return data, errors.Wrap(err, "json.Unmarshal")
	} else if data, err = json.MarshalIndent(raw, "", "   "); err != nil {
		return data, errors.Wrap(err, "json.MarshalIndent")
	}
	return data, nil
}

/***********************************************/
//               exp.Dur Methods               //
/***********************************************/

// Dur is used to export "time.Duration" with expvar.
type Dur struct {
	value time.Duration
}

// Value returns the time.Duration of the metric.
func (v *Dur) Value() time.Duration {
	return v.value
}

// String returns a quoted human readable metric.
func (v *Dur) String() string {
	return fmt.Sprintf(`"%v"`, v.value.String())
}

// Set sets the duration for the metric.
func (v *Dur) Set(duration time.Duration) {
	v.value = duration
}

/***********************************************/
//               exp.Time Methods              //
/***********************************************/

// Time is used to export "time.Time" with expvar.
type Time struct {
	value time.Time
}

// Now sets the value to now.
func (v *Time) Now() {
	v.value = time.Now()
}

// Set sets the time to a date.
func (v *Time) Set(date time.Time) {
	v.value = date
}

// Add sets the time to now+duration.
func (v *Time) Add(duration time.Duration) {
	v.value = time.Now().Add(duration)
}

// String -ify the Time metric
func (v *Time) String() string {
	return fmt.Sprintf(`"%v: %v (%v)"`, v.Value().Unix(), v.value.Format(time.UnixDate), v.Until())
}

// Until returns the rounded duration between now and value.
func (v *Time) Until() time.Duration {
	return time.Until(v.value).Round(time.Second)
}

// Value returns the time.Time() value of the metric.
func (v *Time) Value() time.Time {
	return v.value
}

// SplitSubN splits a string by length and return a slice of string blocks.
func SplitSubN(s string, length int) (blocks []string) {
	var sub string
	for i, r := range s {
		sub += string(r)
		if (i+1)%length == 0 {
			blocks = append(blocks, sub)
			sub = ""
		}
	}
	blocks = append(blocks, sub)
	return
}
