package exp

import (
	// standard
	"expvar"
	"testing"
	"time"

	// external
	"github.com/stretchr/testify/assert"
)

func TestDurMethods(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	testtime := 15 * time.Second
	var testVar Dur
	testVar.Set(testtime)
	a.EqualValues(testtime, testVar.Value(), "the value provided must equal the value received")
	a.EqualValues(`"`+testtime.String()+`"`, testVar.String(), "the string representation appears wrong")
}

func TestTimeMethods(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	testtime := time.Now().Add(-15 * time.Second)
	var testVar Time
	testVar.Set(testtime)
	a.EqualValues(testtime, testVar.Value(), "the value provided must equal the value received")
	a.EqualValues(-15*time.Second, testVar.Until(), "Until must return the same difference provided")
	testVar.Add(10 * time.Second)
	a.EqualValues(10*time.Second, testVar.Until(), "Until must return the calulated difference from Now")
	testVar.Now()
	a.EqualValues(0*time.Second, testVar.Until(), "Until must return 0 when set to Now")
	a.Contains(testVar.String(), "0s", "String must contain 0s difference.")
}

func TestGetMap(t *testing.T) {
	t.Parallel()
	var fakeData expvar.String
	a := assert.New(t)
	testMap := GetMap("Rocklin")
	testMap.Set("Sierra", &fakeData)
	fakeData.Set("Alpine")
	a.EqualValues(`"Alpine"`, testMap.Get("Sierra").String())
	// make sure we get the same map back.
	testMap = GetMap("Rocklin")
	a.EqualValues(`"Alpine"`, testMap.Get("Sierra").String())
}

func TestGetPublishedMap(t *testing.T) {
	t.Parallel()
	var fakeData expvar.String
	a := assert.New(t)
	testMap := GetPublishedMap("Redwood")
	testMap.Set("Sierra", &fakeData)
	fakeData.Set("Menlow")
	a.EqualValues(`"Menlow"`, testMap.Get("Sierra").String())
	// make sure we get the same map back.
	testMap = GetPublishedMap("Redwood")
	a.EqualValues(`"Menlow"`, testMap.Get("Sierra").String())
}

func TestDebug(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	DebugMap := GetMap(DebugMap)
	DebugLog = true
	timeKey := Debug("something interesting may have happened")
	// There is no way to get the exact string back, but this is enough to make sure it's there.
	a.Contains(DebugMap.String(), "something interesting may have happened")
	a.EqualValues("\"something interesting may have happened\"", DebugMap.Get(timeKey).String())
	a.Contains(DebugMap.String(), timeKey)

	timeKey = Debug("wild ride ahead")
	DebugLog = false
	a.EqualValues("\"wild ride ahead\"", DebugMap.Get(timeKey).String())
}

func TestMakePrettyJSONString(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	// These must match, and match the indent level in the function we're testing.
	jsonDataUgly := `{"GetUsers": {"Cached": 6,"Called": 1,"Failed": 0,"Missed": 0,
      "Updated": "1517365528: Wed Jan 31 02:25:28 UTC 2018 (-41m58s)","Users": 2129}}`
	jsonDataPurdy := `{
   "GetUsers": {
      "Cached": 6,
      "Called": 1,
      "Failed": 0,
      "Missed": 0,
      "Updated": "1517365528: Wed Jan 31 02:25:28 UTC 2018 (-41m58s)",
      "Users": 2129
   }
}`
	// Test a valid conversion and its output.
	out, err := MakePrettyJSONString(jsonDataUgly)
	a.Nil(err, "valid json must parse properly")
	a.EqualValues(jsonDataPurdy, string(out), "formatted string seems incorrect")

	// invalid json should return what we provided and an error.
	jsonDataUgly += "," // now it's invalid.
	out, err = MakePrettyJSONString(jsonDataUgly)
	a.NotNil(err, "invalid json must produce an error")
	a.EqualValues(jsonDataUgly, string(out), "invalid json must return what was provided")
}

func TestGetMapList(t *testing.T) {
	//	cannot be parallel because it resets the maps
	a := assert.New(t)
	maps = Maps{}
	a.Nil(GetMapList(), "the maps list is empty and must return nil")
	fakeMaps := []string{"foo", "bar", "baz"}
	for i := range fakeMaps {
		GetMap(fakeMaps[i])
	}
	mapList := GetMapList()
	for _, fake := range fakeMaps {
		a.EqualValues(true, StringInList(fake, mapList))
	}
}

func TestGetMapValuesString(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	var str1, str2, str3 expvar.String
	var int1, int2, int3 expvar.Int
	str1.Set("test string 1")
	str1.Set("test string two")
	str1.Set("test string tres")
	int1.Set(32)
	int2.Set(64)
	int3.Set(256)
	testMap := GetMap("TestGetMapValuesString")
	testMap.Set("key_string1", &str1)
	testMap.Set("key_string2", &str2)
	testMap.Set("key_string3", &str3)
	testMap.Set("key_int1", &int1)
	testMap.Set("key_int2", &int2)
	testMap.Set("key_int3", &int3)
	mapString, err := GetMapValuesString("TestGetMapValuesString")
	a.Nil(err, "there should be no unmarshal errors here")
	a.EqualValues(str1.Value(), mapString["key_string1"])
	a.EqualValues(str2.Value(), mapString["key_string2"])
	a.EqualValues(str3.Value(), mapString["key_string3"])
	a.EqualValues(int1.String(), mapString["key_int1"])
	a.EqualValues(int2.String(), mapString["key_int2"])
	a.EqualValues(int3.String(), mapString["key_int3"])
}

func TestSplitSubN(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	testString := "abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
	splitString := SplitSubN(testString, 10)
	a.EqualValues(62, len(testString), "If this breaks, Go has a bug or you changed the test string.")
	a.EqualValues(7, len(splitString), "A 62 character length string must have 7 items in a list when split every 10 characters.")
	a.EqualValues("YZ", splitString[6], "The 7th item in the list must contain the last two characters in the test string.")
}

// StringInList returns true if the string is in the slice of strings.
func StringInList(FindMe string, TheList []string) bool {
	for _, l := range TheList {
		if FindMe == l {
			return true
		}
	}
	return false
}
