package jira

import (
	"context"
	"crypto/x509"
	"encoding/pem"
	"github.com/pkg/errors"
	"io"
	"io/ioutil"
	"strings"

	goJira "github.com/andygrunwald/go-jira"
	"github.com/dghubble/oauth1"
)

// GetClient returns a JIRA client interface for the given config
func GetClient(config *Config) (Client, error) {
	switch err := config.getJIRAClient(); {
	case err != nil:
		return nil, err
	case config.jira == nil:
		return nil, errors.New("jira client missing")
	}
	return config, nil
}

// GetTicket returns a Jira task.
func (c *Config) GetTicket(ticketKey string) (*Ticket, error) {
	issue, _, err := c.jira.Issue.Get(ticketKey, nil)
	if err != nil {
		return nil, errors.Wrap(err, "jira.Issue.Get")
	}

	jiraTicket := &Ticket{}
	c.mapIssueFromJIRA(jiraTicket, issue)

	return jiraTicket, nil
}

// GetOpenTickets returns a list of open tasks for a project.
func (c *Config) GetOpenTickets(projectKey string) ([]*Ticket, error) {
	findStatus := "status!=Closed and status!=Done and status!=Cancelled and status!=Fixed"
	issues, _, err := c.jira.Issue.Search("project= "+projectKey+" and "+findStatus, nil)
	if err != nil {
		return nil, errors.Wrap(err, "jira.Issue.Search")
	}
	var tickets = make([]*Ticket, 0)

	for _, issue := range issues {
		jiraTicket := &Ticket{}
		c.mapIssueFromJIRA(jiraTicket, &issue)
		tickets = append(tickets, jiraTicket)
	}

	return tickets, nil
}

// CreateTicket makes a new Jira Task
func (c *Config) CreateTicket(jiraTicket *Ticket) error {
	issue, _, err := c.jira.Issue.Create(&goJira.Issue{
		Fields: &goJira.IssueFields{
			Type: goJira.IssueType{
				Name: jiraTicket.IssueType,
			},
			Project: goJira.Project{
				Key: jiraTicket.ProjectKey,
			},
			Summary:     jiraTicket.Summary,
			Description: jiraTicket.Description,
		},
	})

	if err != nil {
		return errors.Wrap(err, "jira.Issue.Create")
	}

	issue, _, err = c.jira.Issue.Get(issue.ID, nil)
	if err != nil {
		return errors.Wrap(err, "jira.Issue.Get")
	}

	c.mapIssueFromJIRA(jiraTicket, issue)
	return nil
}

// AttachFile puts an attachment on a Jira task.
func (c *Config) AttachFile(ticketKey string, reader io.Reader, attachmentName string) error {
	_, _, err := c.jira.Issue.PostAttachment(ticketKey, reader, attachmentName)
	return err
}

// TransitionTicket moves a ticket to a new status like Done.
func (c *Config) TransitionTicket(ticketKey string, transitionID string) error {
	_, err := c.jira.Issue.DoTransition(ticketKey, transitionID)
	return err
}

func (c *Config) mapIssueFromJIRA(ticket *Ticket, issue *goJira.Issue) {
	ticket.TicketKey = issue.Key

	baseURL := c.jira.GetBaseURL()
	host := baseURL.Hostname()
	ticket.URL = "https://" + host + "/browse/" + issue.Key

	if issue.Fields != nil {
		ticket.Status = issue.Fields.Status.Name
		ticket.IssueType = issue.Fields.Type.Name
		ticket.ProjectKey = issue.Fields.Project.Key
		ticket.Summary = issue.Fields.Summary
		ticket.Description = issue.Fields.Description
	}
}

// JIRA CLIENT OAUTH CREATION STUFF - frankensteined from here https://gist.github.com/Lupus/edafe9a7c5c6b13407293d795442fe67

func (c *Config) getJIRAClient() error {
	ctx := context.Background()
	var oauthConfig *oauth1.Config

	if privateKeyBytes, err := ioutil.ReadFile(c.JiraPrivateKeyLocation); err != nil {
		return errors.Wrap(err, "reading private key file")
	} else if keyDERBlock, _ := pem.Decode(privateKeyBytes); keyDERBlock == nil {
		return errors.Errorf("error decoding key PEM block in %v", c.JiraPrivateKeyLocation)
	} else if !(keyDERBlock.Type == "PRIVATE KEY" || strings.HasSuffix(keyDERBlock.Type, " PRIVATE KEY")) {
		return errors.Errorf("unexpected key DER block type %v", keyDERBlock.Type)
	} else if privateKey, err := x509.ParsePKCS1PrivateKey(keyDERBlock.Bytes); err != nil {
		return errors.Wrap(err, "parsing PKCS1 private key")
	} else {
		oauthConfig = &oauth1.Config{
			ConsumerKey: c.JiraConsumerKey,
			Signer:      &oauth1.RSASigner{PrivateKey: privateKey},
		}
	}

	httpClient := oauth1.NewClient(ctx, oauthConfig, oauth1.NewToken(c.JiraAccessToken, c.JiraAccessTokenSecret))
	newClient, err := goJira.NewClient(httpClient, c.JiraBaseURL)
	if err != nil {
		return errors.Wrap(err, "creating new Jira Client")
	}
	c.jira = newClient
	return nil
}
