package pagerduty

import (
	// external
	goPd "github.com/PagerDuty/go-pagerduty"
)

type client struct {
	pdClient *goPd.Client
}

// GetClient returns a Pagerduty client interface for the given api token
func GetClient(apiToken string) Client {
	return &client{
		pdClient: goPd.NewClient(apiToken),
	}
}

func (c *client) ListServices() ([]Service, error) {
	opts := goPd.ListServiceOptions{}
	opts.Limit = 100

	serviceList := make([]Service, 0)

	for {
		listServicesResponse, err := c.pdClient.ListServices(opts)
		if err != nil {
			return nil, err
		}
		for _, svc := range listServicesResponse.Services {
			serviceList = append(serviceList, mapFromPdService(svc))
		}
		if !listServicesResponse.More {
			break
		}
		opts.Offset = listServicesResponse.Limit + listServicesResponse.Offset
	}

	return serviceList, nil
}

func (c *client) ListEscalationPolicies() ([]EscalationPolicy, error) {
	opts := goPd.ListEscalationPoliciesOptions{}
	opts.Limit = 100

	escalationPolicyList := make([]EscalationPolicy, 0)

	for {
		listEscalationPoliciesResponse, err := c.pdClient.ListEscalationPolicies(opts)
		if err != nil {
			return nil, err
		}
		for _, ep := range listEscalationPoliciesResponse.EscalationPolicies {
			escalationPolicyList = append(escalationPolicyList, mapFromPdEscalationPolicy(ep))
		}
		if !listEscalationPoliciesResponse.More {
			break
		}
		opts.Offset = listEscalationPoliciesResponse.Limit + listEscalationPoliciesResponse.Offset
	}

	return escalationPolicyList, nil
}

func (c *client) GetIntegration(serviceID string, integrationID string) (Integration, error) {
	integration, err := c.pdClient.GetIntegration(serviceID, integrationID, goPd.GetIntegrationOptions{})
	if err != nil {
		return Integration{}, err
	}

	return Integration{
		ID:               integration.ID,
		Type:             integration.Type,
		IntegrationEmail: integration.IntegrationEmail,
	}, nil
}

func (c *client) ListOnCalls() ([]OnCall, error) {
	opts := goPd.ListOnCallOptions{}
	opts.Limit = 100

	oncallsList := make([]OnCall, 0)

	for {
		oncallsResponse, err := c.pdClient.ListOnCalls(opts)
		if err != nil {
			return nil, err
		}
		for _, oc := range oncallsResponse.OnCalls {
			oncallsList = append(oncallsList, mapFromPdOnCall(oc))
		}
		if !oncallsResponse.More {
			break
		}
		opts.Offset = oncallsResponse.Limit + oncallsResponse.Offset
	}

	return oncallsList, nil
}

func mapFromPdOnCall(pdOnCall goPd.OnCall) OnCall {
	return OnCall{
		EscalationLevel:  pdOnCall.EscalationLevel,
		EscalationPolicy: mapFromPdAPIObject(pdOnCall.EscalationPolicy),
		UserID:           pdOnCall.User.ID,
		UserType:         pdOnCall.User.Type,
		UserSummary:      pdOnCall.User.Summary,
		ScheduleID:       pdOnCall.Schedule.ID,
	}
}

func mapFromPdEscalationPolicy(pdEscalationPolicy goPd.EscalationPolicy) EscalationPolicy {
	ep := EscalationPolicy{
		ID:      pdEscalationPolicy.ID,
		Summary: pdEscalationPolicy.Summary,
	}
	if pdEscalationPolicy.Services == nil {
		return ep
	}

	ep.Services = make([]ObjectReference, 0)

	for _, svc := range pdEscalationPolicy.Services {
		ep.Services = append(ep.Services, mapFromPdAPIReference(svc))
	}

	return ep
}

func mapFromPdService(pdService goPd.Service) Service {
	svc := Service{
		ID:          pdService.ID,
		Name:        pdService.Name,
		Description: pdService.Description,
	}

	if pdService.Integrations == nil {
		return svc
	}

	svc.Integrations = make([]ObjectReference, 0)

	for _, integration := range pdService.Integrations {
		// Although the PD client thinks these are integration objects, in reality they are just object references
		svc.Integrations = append(svc.Integrations, mapObjectReferenceFromIntegration(integration))
	}

	return svc
}

func mapObjectReferenceFromIntegration(psIntegration goPd.Integration) ObjectReference {
	return ObjectReference{
		ID:   psIntegration.ID,
		Type: psIntegration.Type,
	}
}

// I don't know why they have 2 different things that seem to do the same thing :(
func mapFromPdAPIObject(pdAPIObject goPd.APIObject) ObjectReference {
	return ObjectReference{
		ID:      pdAPIObject.ID,
		Summary: pdAPIObject.Summary,
	}
}

// I don't know why they have 2 different things that seem to do the same thing :(
func mapFromPdAPIReference(pdAPIReference goPd.APIReference) ObjectReference {
	return ObjectReference{
		ID:   pdAPIReference.ID,
		Type: pdAPIReference.Type,
	}
}
