package pagerduty

import (
	// standard
	"testing"

	// external
	goPd "github.com/PagerDuty/go-pagerduty"
	"github.com/stretchr/testify/assert"
)

func TestMapOnCallObject(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	// We're only testing the values we actually use in the returned object
	pdOncall := goPd.OnCall{
		EscalationLevel: 3,
		User: goPd.APIObject{
			Summary: "USERSUMMARY",
		},
		EscalationPolicy: goPd.APIObject{
			ID:      "EPID",
			Summary: "EPSUMMARY",
		},
		Schedule: goPd.APIObject{
			ID: "SCHEDULEID",
		},
	}
	mappedOnCall := mapFromPdOnCall(pdOncall)

	a.EqualValues(3, mappedOnCall.EscalationLevel)
	a.EqualValues("USERSUMMARY", mappedOnCall.UserSummary)
	a.EqualValues("EPID", mappedOnCall.EscalationPolicy.ID)
	a.EqualValues("EPSUMMARY", mappedOnCall.EscalationPolicy.Summary)
	a.EqualValues("SCHEDULEID", mappedOnCall.ScheduleID)
}

func TestMapEscalationPolicyObject(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	// We're only testing the values we actually use in the returned object
	pdEscalationPolicy := goPd.EscalationPolicy{
		APIObject: goPd.APIObject{
			ID:      "EPID",
			Summary: "EPSUMMARY",
		},
		Services: []goPd.APIReference{
			goPd.APIReference{
				ID:   "SERVICE1ID",
				Type: "SERVICE1TYPE",
			},
			goPd.APIReference{
				ID:   "SERVICE2ID",
				Type: "SERVICE2TYPE",
			},
		},
	}
	mappedEscalationPolicy := mapFromPdEscalationPolicy(pdEscalationPolicy)

	a.EqualValues("EPID", mappedEscalationPolicy.ID)
	a.EqualValues("EPSUMMARY", mappedEscalationPolicy.Summary)
	a.EqualValues(2, len(pdEscalationPolicy.Services))
	a.EqualValues("SERVICE1ID", pdEscalationPolicy.Services[0].ID)
	a.EqualValues("SERVICE1TYPE", pdEscalationPolicy.Services[0].Type)
	a.EqualValues("SERVICE2ID", pdEscalationPolicy.Services[1].ID)
	a.EqualValues("SERVICE2TYPE", pdEscalationPolicy.Services[1].Type)

	// Test empty services lists
	pdEscalationPolicy.Services = nil

	mappedEscalationPolicy = mapFromPdEscalationPolicy(pdEscalationPolicy)
	a.Nil(pdEscalationPolicy.Services)
}

func TestMapServiceObject(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	// We're only testing the values we actually use in the returned object
	pdService := goPd.Service{
		APIObject: goPd.APIObject{
			ID: "SERVICEID",
		},
		Name: "SERVICENAME",
		Integrations: []goPd.Integration{
			goPd.Integration{
				APIObject: goPd.APIObject{
					ID: "INTEGRATION1ID",
				},
				Type: "INTEGRATION1TYPE",
			},
			goPd.Integration{
				APIObject: goPd.APIObject{
					ID: "INTEGRATION2ID",
				},
				Type: "INTEGRATION2TYPE",
			},
		},
	}
	mappedService := mapFromPdService(pdService)

	a.EqualValues("SERVICEID", mappedService.ID)
	a.EqualValues("SERVICENAME", mappedService.Name)
	a.EqualValues(2, len(mappedService.Integrations))
	a.EqualValues("INTEGRATION1ID", mappedService.Integrations[0].ID)
	a.EqualValues("INTEGRATION1TYPE", mappedService.Integrations[0].Type)
	a.EqualValues("INTEGRATION2ID", mappedService.Integrations[1].ID)
	a.EqualValues("INTEGRATION2TYPE", mappedService.Integrations[1].Type)

	// Test empty integrations lists
	pdService.Integrations = nil

	mappedService = mapFromPdService(pdService)
	a.Nil(mappedService.Integrations)
}
