package slack

import (
	// standard
	"expvar"
	"sync"

	// external
	"github.com/nlopes/slack"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
)

// Allow importing packages to find channels and users easily.
const (
	ChannelRegex       = `^<#((?:C)[^\|]*)\|.*>$`
	DirectMessageRegex = `^<#((?:D)[^\|]*)\|.*>$`
	GroupRegex         = `^<#((?:G)[^\|]*)\|.*>$`
	UserRegex          = `^<@(U.*)>$`
)

// Config for the JIRA package; some from TOML, some created for each instance.
type Config struct {
	APIToken     string `toml:"api_token"`
	DebugChannel string `toml:"debug_channel"`
	LogDir       string `toml:"log_dir"`
	Hostname     string
	BotName      string
	slack        *slack.Client
	responseChan chan Response // Used to send slack replies via RTM.
	stopChan     chan bool     // Used to stop RTM.
	users        slackUsers    // Caches the Slack User list.
	chans        slackChans    // Caches the Slack Chan list.
	groups       slackGroups   // Caches the Slack Group list.
	export       expvarData    // Contains exported expvar variables.
}

// Client interface represents available functions for interacting with Slack
// If any of this changes, counterfeiter must be re-run.
type Client interface {
	Dial(handlers []Handler) *expvar.Map
	Stop()
	SendReply(msg Message, reply string) Response
	SendMessage(RespondTo string, Recipient []string, Message string) Response
	SendAttachments(RespondTo string, msg string, Attachments []slack.Attachment) Response
	GetEntityName(entityID string) string
	GetUserDMChannelID(userID string) (string, error)
	ReturnLogDir(appendage string) string
}

// Handler is an interface that represents required methods for a glitch command, we separate the logic of IsMatch and Execute to ensure that they are considered as separate operations
// It's important that IsMatch is as lightweight as possible as it's called for every message that glitch observes
type Handler interface {
	IsMatch(msg Message) bool  // Called on every line glitch sees to determine whether the command should be executed
	Execute(msg Message) error // Called in a separate goroutine if IsMatch returns true
	Stop() error               // Called to stop any running goroutines.
	Export() *expvar.Map       // Returns the handler's expvar Map.
	Name() string              // Returns the handler's package name.
}

// Response is used to input a message into this library.
type Response struct {
	Recipient   []string
	Message     string
	RespondTo   string
	Attachments []slack.Attachment
}

// Message defines the content to send a message to Slack.
type Message struct {
	Data      slack.Msg
	RespondTo string
	User      string
	IsAdmin   bool
}

// This stores the list of slack channels in memory for easy access.
type slackChans struct {
	List    []slack.Channel
	Count   expvar.Int
	Updated exp.Time
	sync.Mutex
}

// And this stores the user list.
type slackUsers struct {
	List    []slack.User
	Count   expvar.Int
	Updated exp.Time
	sync.Mutex
}

// This stores the group list.
type slackGroups struct {
	List    []slack.Group
	Count   expvar.Int
	Updated exp.Time
	sync.Mutex
}

// expvarData is used to export data through expvar.
type expvarData struct {
	slackData     *expvar.Map
	cmdsMatched   expvar.Int
	failedExecs   expvar.Int
	debugChan     expvar.String
	defaultHit    expvar.Int
	slackHealth   expvar.String
	slackFirst    exp.Time
	slackLast     exp.Time
	slackConnects expvar.Int
	slackFails    expvar.Int
	slackLatency  exp.Dur
	slackID       expvar.String
}
