package slack

import (
	// standard
	"expvar"
	"log"
	"path"
	"strings"
	"time"

	// external
	goslack "github.com/nlopes/slack"

	// local

	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
)

// GetClient generates a slack client interface using the supplied config
func GetClient(config *Config) Client {
	config.slack = goslack.New(config.APIToken)
	config.responseChan = make(chan Response)
	config.stopChan = make(chan bool, 1)
	return config
}

// Dial connects to slack and starts a message loop.
func (c *Config) Dial(handlers []Handler) *expvar.Map {
	rtm := c.slack.NewRTM()
	go rtm.ManageConnection()
	c.exportData()
	go c.watchSlack(*rtm, handlers)
	return c.export.slackData
}

// Stop shuts down the MessageLoop routine and disconnects from Slack.
func (c *Config) Stop() {
	c.stopChan <- true
}

// ReturnLogDir provides a directory in which to make logs.
func (c *Config) ReturnLogDir(appendage string) string {
	if appendage != "" {
		return path.Join(c.LogDir, appendage)
	}
	return c.LogDir
}

// Get internal counters and health checks in place for expvar c.export.
func (c *Config) exportData() {
	userData := exp.GetMap("slack.GetUsers")
	chanData := exp.GetMap("slack.GetChannels")
	groupData := exp.GetMap("slack.GetGroups")
	healthData := exp.GetMap("slack.Health")
	messageData := exp.GetMap("slack.Messages")
	c.export.slackData = exp.GetMap("slack.Data")

	messageData.Set("handler_matches", &c.export.cmdsMatched)
	messageData.Set("handler_failues", &c.export.failedExecs)

	c.export.debugChan.Set(c.DebugChannel)
	c.export.slackData.Set("Messages", messageData)
	c.export.slackData.Set("GetUsers", userData)
	c.export.slackData.Set("GetChannels", chanData)
	c.export.slackData.Set("GetGroups", groupData)
	c.export.slackData.Set("Health", healthData)

	healthData.Set("ConnectionID", &c.export.slackID)
	healthData.Set("DebugChannel", &c.export.debugChan)
	healthData.Set("Status", &c.export.slackHealth)
	healthData.Set("DefaultCaseCount", &c.export.defaultHit)
	healthData.Set("FirstConnected", &c.export.slackFirst)
	healthData.Set("LastConnected", &c.export.slackLast)
	healthData.Set("SuccessfulConnects", &c.export.slackConnects)
	healthData.Set("FailedConnects", &c.export.slackFails)
	healthData.Set("ServerLatency", &c.export.slackLatency)

	userData.Set("Users", &c.users.Count)
	userData.Set("Updated", &c.users.Updated)

	chanData.Set("Channels", &c.chans.Count)
	chanData.Set("Updated", &c.chans.Updated)

	groupData.Set("Groups", &c.groups.Count)
	groupData.Set("Updated", &c.groups.Updated)
}

// watchSlack processes every message received from slack api.
func (c *Config) watchSlack(rtm goslack.RTM, handlers []Handler) {
	for {
		select {
		//Handle & filter incomming messages on the channel.
		case msg := <-rtm.IncomingEvents:

			switch ev := msg.Data.(type) {
			case *goslack.ConnectedEvent:
				c.export.slackID.Set(ev.Info.User.ID)
				c.export.slackHealth.Set("connected")
				c.export.slackLast.Now()
				if c.export.slackConnects.Add(1); c.export.slackConnects.Value() == 1 {
					c.export.slackFirst.Now()
				}
				log.Println("Connected to Slack, ID:", c.export.slackID.Value())

			case *goslack.HelloEvent:
				go func() {
					resp := ("*First Connected at:* " + c.export.slackFirst.Value().Format(time.RFC3339) +
						"\n*User ID:* " + c.export.slackID.Value() +
						"\n*Glitch Version:* " + strings.Replace(exp.GetPublishedMap(c.BotName).Get("glitch_version").String(), `"`, ``, -1))
					if c.export.slackConnects.Value() > 1 {
						resp += "\n*Re-Connected at:* " + c.export.slackLast.Value().Format(time.RFC3339) + "\n:bleedPurple:"
					}
					resp += "\n*Hostname or URL*: " + c.Hostname
					rtm.SendMessage(rtm.NewOutgoingMessage(resp, c.export.debugChan.Value()))
				}()
				exp.Debug("Hello Event")

			case *goslack.MessageEvent:
				// Don't worry about messages from self
				if ev.Msg.User != c.export.slackID.Value() {
					// Kick this off so we can keep looking for more commands.
					go c.CheckMessage(Message{
						Data:      ev.Msg,
						RespondTo: ev.Msg.Channel,
						User:      ev.Msg.User,
						IsAdmin:   ev.Msg.Channel == c.export.debugChan.Value(),
					}, handlers)
				}

			case *goslack.LatencyReport:
				c.export.slackLatency.Set(ev.Value)

			case *goslack.RTMError:
				c.export.slackFails.Add(1)
				log.Println("Error:", ev.Error())
				c.export.slackHealth.Set("disconnected? " + ev.Error())

			case *goslack.InvalidAuthEvent:
				c.export.slackFails.Add(1)
				c.export.slackHealth.Set("disconnected, invalid credentials, no longer trying")
				log.Print("Error: Invalid credentials - Slack Connection Failed!")
				c.stopChan <- true

			case *goslack.ConnectionErrorEvent:
				c.export.slackFails.Add(1)
				c.export.slackHealth.Set("disconnected, error connecting, still trying")
				log.Println("Error Connecting to Slack. Invoking backoff timer.")

			case *goslack.PresenceChangeEvent:
				log.Println("Presence Change:", ev)

			case *goslack.OutgoingErrorEvent:
				log.Printf("Outgoing Error: %v\n", ev.Error())

			case *goslack.MessageTooLongEvent:
				log.Printf("Error: %v\n", ev.Error())

			case *goslack.UserTypingEvent:
				// ignore user's typing.

			case *goslack.RateLimitEvent:
				log.Printf("Error: %v\n", ev.Error())

			case *goslack.UserChangeEvent:
				// don't care about these either.

			case *goslack.RTMEvent:
				log.Println("RTM Event:", msg.Data)

			default:
				// log.Printf("Unexpected: %v\n", msg.Data)
				c.export.defaultHit.Add(1)
				// If data is strange then do nothing.
			}

			// Handle outgoing messages by checking the response channel.
		case response := <-c.responseChan:
			// If the attachment has any of these fields, it's real, send it.
			if a := response.Attachments; a != nil {
				postParams := goslack.PostMessageParameters{AsUser: true}
				for _, attachment := range a {
					postParams.Attachments = append(postParams.Attachments, attachment)
				}
				if _, _, err := rtm.PostMessage(response.RespondTo, response.Message, postParams); err != nil {
					log.Println("Error From rtm.PostMessage:", err)
				}
			} else if reply := response.Message; reply != "" {
				for _, recipient := range response.Recipient {
					if recipient != "" {
						reply = "<@" + recipient + "> " + reply
					}
				}
				rtm.SendMessage(rtm.NewOutgoingMessage(reply, response.RespondTo))
			}

		case <-c.stopChan:
			log.Print("Disconnecting from Slack and stopping MessageLoop.")
			if err := rtm.Disconnect(); err != nil {
				log.Println("Disconnecting from Slack:", err)
			}
			return
		}
	}
}

// SendReply is a helper function for quick text-only replies to specific messages
func (c *Config) SendReply(msg Message, reply string) Response {
	return c.SendMessage(msg.RespondTo, []string{msg.Data.User}, reply)
}

// SendMessage to someone over Slack.
func (c *Config) SendMessage(RespondTo string, Recipient []string, Message string) Response {
	resp := Response{Recipient: Recipient, RespondTo: RespondTo, Message: Message}
	c.responseChan <- resp
	return resp
}

// SendAttachments to someone over Slack.
func (c *Config) SendAttachments(RespondTo string, msg string, Attachments []goslack.Attachment) Response {
	resp := Response{RespondTo: RespondTo, Message: msg, Attachments: Attachments}
	c.responseChan <- resp
	return resp
}

// CheckMessage fires a handler for a Matched command.
func (c *Config) CheckMessage(msg Message, handlers []Handler) {
	for _, handler := range handlers {
		if handler.IsMatch(msg) {
			c.export.cmdsMatched.Add(1)
			if err := handler.Execute(msg); err != nil {
				c.export.failedExecs.Add(1)
				log.Println("Error executing handler", handler.Name(), "-", err)
				c.SendMessage(msg.RespondTo, []string{msg.Data.User}, "Error executing handler: "+err.Error())
			}
		}
	}
}

// GetEntityName returns the readable name from the slack entity ID, supports users, channels, groups and IMs
func (c *Config) GetEntityName(entityID string) string {
	if strings.HasPrefix(entityID, "C") {
		return c.getChannelName(entityID)
	}

	if strings.HasPrefix(entityID, "G") {
		return c.getGroupName(entityID)
	}

	if strings.HasPrefix(entityID, "U") {
		return c.getUserName(entityID)
	}

	if strings.HasPrefix(entityID, "D") {
		// IMs don't really have names, so we're just going to use the ID for IMs
		return entityID
	}

	return "<error_generating_entity_name>"
}

// getChannelName returns the channel name for the given ID
func (c *Config) getChannelName(channelID string) string {
	if c.chans.List != nil {
		for _, chn := range c.chans.List {
			if chn.ID == channelID {
				return chn.Name
			}
		}
	}

	var err error
	c.chans.Lock()
	defer c.chans.Unlock()

	if c.chans.List, err = c.slack.GetChannels(true); err != nil {
		return "<error_getting_channel_name>"
	}
	c.chans.Updated.Now()
	c.chans.Count.Set(int64(len(c.chans.List)))
	for _, chn := range c.chans.List {
		if chn.ID == channelID {
			return chn.Name
		}
	}
	return "<channel_name_not_found>"
}

// getUserName returns a username if it can be found.
func (c *Config) getUserName(userID string) string {
	if c.users.List != nil {
		for _, user := range c.users.List {
			if user.ID == userID {
				return user.Name
			}
		}
	}

	var err error
	c.users.Lock()
	defer c.users.Unlock()

	if c.users.List, err = c.slack.GetUsers(); err != nil {
		return "<error_getting_user_name>"
	}
	c.users.Updated.Now()
	c.users.Count.Set(int64(len(c.users.List)))
	for _, user := range c.users.List {
		if user.ID == userID {
			return user.Name
		}
	}

	return "<user_name_not_found>"
}

// getGroupName returns the group name for the given ID
func (c *Config) getGroupName(groupID string) string {
	if c.groups.List != nil {
		for _, grp := range c.groups.List {
			if grp.ID == groupID {
				return grp.Name
			}
		}
	}

	var err error
	c.groups.Lock()
	defer c.groups.Unlock()

	if c.groups.List, err = c.slack.GetGroups(true); err != nil {
		return "<error_getting_group_name>"
	}
	c.groups.Updated.Now()
	c.groups.Count.Set(int64(len(c.groups.List)))
	for _, grp := range c.groups.List {
		if grp.ID == groupID {
			return grp.Name
		}
	}

	return "<group_name_not_found>"
}

// GetUserDMChannelID opens a direct message to the supplied userID and returns the channelID so that messages can be sent there
func (c *Config) GetUserDMChannelID(userID string) (string, error) {
	_, _, channelID, err := c.slack.OpenIMChannel(userID)
	if err != nil {
		return "", err
	}

	return channelID, nil
}
