package main

import (
	"flag"
	"fmt"
	"log"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ecs"
)

type Config struct {
	statsite  *ServiceConfig
	proxy     *ServiceConfig
	awsRegion string
	proxyOnly bool
}

type ServiceConfig struct {
	imageTag    string
	serviceName string
	taskDefArn  string
	clusterName string
}

func parseAndValidateFlags() *Config {
	conf := &Config{
		statsite: &ServiceConfig{},
		proxy:    &ServiceConfig{},
	}

	flag.Usage = func() {
		fmt.Fprintf(os.Stderr, "Deploys the statsite and go-statsd-proxy services, updating go-statsd-proxy with the new statsite hosts.\n")
		fmt.Fprintf(os.Stderr, "The deploy process is: \n\t* register a new task definition(updating the docker image if provided, and setting STATSD_HOSTS env var for the proxy)\n\t* update the service to use that task definition\n\t* wait until there are desired_count number of tasks running the new task definition.\n\n")
		fmt.Fprintf(os.Stderr, "Usage:\n")
		flag.PrintDefaults()
	}

	flag.StringVar(&conf.statsite.imageTag, "statsiteImageTag", "", "The docker image tag to deploy to the statsite service(must be an image contained in the statsite ECR repo in the account being deployed to). If not provided then the current image ID will be used.")
	flag.StringVar(&conf.statsite.taskDefArn, "statsiteTaskDefArn", "", "The ARN of the statsite task definition to base the new task definition off of.")
	flag.StringVar(&conf.statsite.serviceName, "statsiteServiceName", "statsite", "The name of the ECS statsite service.")
	flag.StringVar(&conf.statsite.clusterName, "statsiteClusterName", "statsite", "The name of the ECS statsite cluster.")

	flag.StringVar(&conf.proxy.imageTag, "proxyImageTag", "", "The docker image tag to deploy to the statsd proxy service(must be an image contained in the proxy ECR repo in the account being deployed to). If not provided then the current image ID will be used.")
	flag.StringVar(&conf.proxy.taskDefArn, "proxyTaskDefArn", "", "The ARN of the statsd proxy task definition to base the new task definition off of.")
	flag.StringVar(&conf.proxy.serviceName, "proxyServiceName", "go-statsd-proxy", "The name of the ECS statsd proxy service.")
	flag.StringVar(&conf.proxy.clusterName, "proxyClusterName", "go-statsd-proxy", "The name of the ECS statsd proxy cluster.")

	flag.BoolVar(&conf.proxyOnly, "proxyOnly", false, "Only deploy statsd proxy with updated hosts. Does not deploy statsite.")
	flag.StringVar(&conf.awsRegion, "awsRegion", "us-west-2", "The region that the ECS resources are located in.")
	flag.Parse()

	if conf.statsite.serviceName == "" || conf.statsite.clusterName == "" || (conf.statsite.taskDefArn == "" && !conf.proxyOnly) {
		log.Fatal("statsite clusterName, and taskDefArn are required arguments.")
		return nil
	}

	if conf.proxy.serviceName == "" || conf.proxy.clusterName == "" || conf.proxy.taskDefArn == "" {
		log.Fatal("proxy serviceName, clusterName, and taskDefArn are required arguments.")
		return nil
	}

	return conf
}

func check(err error) {
	if err != nil {
		log.Fatal(err)
		os.Exit(1)
	}
}

func main() {
	conf := parseAndValidateFlags()

	sess, err := session.NewSession(&aws.Config{
		Region:              aws.String(conf.awsRegion),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	})
	check(err)
	ecsClient := ecs.New(sess)
	ec2Client := ec2.New(sess)

	// create new statsite task def with new image ID if needed
	var statsiteTaskDefArn string
	if !conf.proxyOnly {
		statsiteTaskDefArn, err = updateStatsite(ecsClient, conf.statsite)
		check(err)
	} else {
		out, err := ecsClient.DescribeServices(&ecs.DescribeServicesInput{
			Cluster:  &conf.statsite.clusterName,
			Services: []*string{&conf.statsite.serviceName},
		})
		check(err)
		statsiteTaskDefArn = *out.Services[0].TaskDefinition
	}

	statsiteHosts, err := getStatsiteHosts(ecsClient, ec2Client, conf.statsite, statsiteTaskDefArn)
	check(err)

	// create new go-statsd-proxy task def w/ updated statsite hosts
	err = updateStatsdProxy(ecsClient, conf.proxy, statsiteHosts)
	check(err)

	log.Println("All done!")
}
