import logging
import dns.resolver
import boto3
import json
import sys
import os

# Logging (replace default)
logger = logging.getLogger()
if logger.handlers:
    for handler in logger.handlers:
        logger.removeHandler(handler)
logging.basicConfig(stream=sys.stdout, level=logging.INFO)

# Lambda Envrionment
RETRY_QUEUE = os.environ.get('RETRY_QUEUE')
AWS_REGION = os.environ.get('AWS_REGION')

# hardcode Environment (can make smarter with lambda envs in the future if we have infoblox use this too)
DOMAIN = os.environ.get('TARGET_DNS_DOMAIN')
LOOKUP_DOMAIN = os.environ.get('LOOKUP_DNS_DOMAIN')
R53_ASSUME = os.environ.get('R53_ASSUME_ROLE')
ALLOWED_ROLES = '|'.split(os.environ.get('ALLOW_LIST_MACHINE_CLASSES'))

# Instantiate clients
sts = boto3.client('sts', region_name=AWS_REGION, endpoint_url="https://sts.{}.amazonaws.com".format(AWS_REGION))
sqs = boto3.client('sqs', region_name=AWS_REGION)

# Determine what stage of our "state machine" we're in; return proper signals
def sm_determine(hostname, message, source):
    # If this message is being invoked from the retry queue, just fail into retry DLQ
    if 'delay' in source:
        logger.info(" Retry for %s failed. Putting into DLQ", hostname)
        return False
    else:
        # send message to retry queue
        try:
            sqs.send_message(QueueUrl=RETRY_QUEUE, MessageBody=message)
        except Exception as e:  # I don't know what error codes there are :D 
            logger.error(" Error sending message for %s to delay queue: %s", hostname, e)
            return False
        # Identify message as "successfully" sent to the next stage of the workflow (retry queue)
        return True


# Upsert the route 53 IVS name.
# Return True if update was successful. Return False otherwise
# NOTE: Any errors with this function will result in the lambda exiting without going through the retry workflow
def update_dns(contents, source):
    # form subdomain
    subdomain = "{}.{}".format(contents.get('pop'), DOMAIN)
    # Connectivity and creds
    try:
        # Assume that role
        sts_creds = sts.assume_role(
            RoleArn=R53_ASSUME,
            RoleSessionName="R53CrudSession"
        )
        r53 = boto3.client(
            'route53',
            aws_access_key_id=sts_creds.get('Credentials', {}).get('AccessKeyId'),
            aws_secret_access_key=sts_creds.get('Credentials', {}).get('SecretAccessKey'),
            aws_session_token=sts_creds.get('Credentials', {}).get('SessionToken'),
        )
    except Exception as e:
        logger.error(" Failed Route53 connection steps: %s", e)
        return False
    # get R53 Zone ID
    try:
        # thanks christi LOL
        hosted_zone = r53.list_hosted_zones_by_name(DNSName=subdomain, MaxItems="1")
        hosted_zone_id = hosted_zone.get('HostedZones')[0].get('Id').split('/')[2]
    except Exception as e:
        logger.error(" Failed to find zone ID for %s: %s", subdomain, e)
        return False  # weird error so don't even bother retrying. Directly to normal DLQ
    # upsert record
    ivs_fqdn = "{}.{}".format(contents.get('hostname').split('.')[0], subdomain)
    try:
        r53.change_resource_record_sets(
            HostedZoneId=hosted_zone_id,
            ChangeBatch={
                "Comment": "HLS CRUD Lambda update beep boop",
                "Changes": [{
                    "Action": "UPSERT",
                    "ResourceRecordSet": {
                        "Name": ivs_fqdn,
                        "Type": "A",
                        "TTL": 300,
                        "ResourceRecords": [{"Value": contents.get('new_ip_address')}]
                    }
                }]
            }
        )
    except Exception as e:
        logger.error(" Failed to upsert %s: %s", ivs_fqdn, e)
        return False
    # Let's log successes to make searching easier
    logger.info(" %s (subdomain %s) upserted successfully with IP %s", ivs_fqdn, subdomain, contents.get('new_ip_address'))
    return True


# Decide how a message is handled depending on its contents
# Return True if message was processed succesfully. Return False otherwise
def handle_message(record, source):
    # load in the message
    try:
        raw_message = json.loads(record)
        # Dumb ternary because the direct SQS SendMessage format is simpler than the SNS -> SQS message format 
        # which wraps the message json into the 'Message' key
        contents = json.loads(raw_message.get('Message')) if 'Message' in raw_message else raw_message
    except json.JSONDecodeError as e:
        logger.error(" Failed to load message json: %s", e)
        return False

    # noop for non-allowed roles (video-edge is only whitelisted)
    try:
        if contents['twitch_role'] not in ALLOWED_ROLES:
            logger.info(" Role %s not allowed.", contents.get('twitch_role'))
            return True
    except KeyError as e:
        logger.error(" Malformed message did not include 'twitch_role' key")
        return False

    # On bootup, dhclient may send a PREINIT message with no IPs. If there's no IP given, let it pass.
    if contents.get('new_ip_address') == '':
        logger.error(" Received an SNS message with an empty IP from %s", contents.get('hostname'))
        return True

    # Find the value for the A record
    try:
        fqdn_pts = '.'.split(contents.get('hostname'))
        lookup = "{}.{}.{}".format(fqdn_pts[0], fqdn_pts[1], LOOKUP_DOMAIN)
        dns_records = dns.resolver.query(lookup)
    except dns.resolver.NXDOMAIN as e:
        logger.error(" Failed to lookup IP for %s: %s", contents.get('hostname'), e)
        return sm_determine(contents.get('hostname'), raw_message.get('Message', '{}'), source)

    # If a result was found, try to match it with the IP provided by the SNS message
    for r in dns_records:
        ip = r.to_text()
        if ip != contents.get('new_ip_address'):
            logger.error(" IP in DNS for %s (%s) does not match message from SNS (%s)", contents.get('hostname'), ip, contents.get('new_ip_address'))
            return sm_determine(contents.get('hostname'), raw_message.get('Message', '{}'), source)
    # Found the DNS, IP matches, so update the DNS
    return update_dns(contents, source)


def lambda_handler(event, context):
    # Through event source mappings, multiple messages may be processed
    for record in event.get('Records'):
        if not handle_message(record.get('body'), record.get('eventSourceARN')):
            # Even if 1 message in the batch fails, fail completely
            # 'All messages in a failed batch return to the queue, so your function code must be able to process the same message multiple times without side effects'
            # https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html
            sys.exit(1)

    logger.info(" Processed %s messages", len(event.get('Records')))
