
import sys

import ldap
from flask import current_app, jsonify, request
from flask_cors import cross_origin

from alerta.auth.utils import create_token, get_customers
from alerta.exceptions import ApiError
from alerta.models.permission import Permission
from alerta.models.user import User
from alerta.utils.audit import auth_audit_trail

from . import auth

# this module requires following parameters in the configuration:
# LDAP_URL (e.g. 'ldaps://ldaps.my.domain')
# LDAP_BASEDN (e.g. 'ou=Users,dc=my,dc=domain')

@auth.route('/auth/login', methods=['OPTIONS', 'POST'])
@cross_origin(supports_credentials=True)
def login():
    # Allow LDAP server to use a self signed certificate
    if current_app.config['LDAP_ALLOW_SELF_SIGNED_CERT']:
        ldap.set_option(ldap.OPT_X_TLS_REQUIRE_CERT, ldap.OPT_X_TLS_ALLOW)

    # Retrieve required fields from client request
    login = request.json.get('username', None) or request.json.get('email')
    password = request.json.get('password')
    if not login or not password:
        raise ApiError("must supply 'username' and 'password'", 401)

    # Search for user
    try:
        trace_level = 2 if current_app.debug else 0
        ldap_connection = ldap.initialize(current_app.config['LDAP_URL'], trace_level=trace_level)
        ldap_connection.simple_bind_s()
        ldap_result = ldap_connection.search_s(current_app.config['LDAP_OPENDJ_BASEDN'], ldap.SCOPE_SUBTREE, 'uid={}'.format(login), [])
    except Exception as e:
        raise ApiError(str(e), 500)

    if len(ldap_result) != 1:
        raise ApiError('Invalid username "{}"'.format(login), 401)

    # Attempt LDAP AUTH
    try:
        userdn = ldap_result[0][0]
        ldap_connection.simple_bind_s(userdn, password)
    except ldap.INVALID_CREDENTIALS:
        raise ApiError('invalid username or password', 401)
    except Exception as e:
        raise ApiError(str(e), 500)

    # Get email address from LDAP
    email_verified = False
    email = ''
    try:
        email = ldap_result[0][1]['mail'][0].decode(sys.stdout.encoding)
        email_verified = True
    except Exception as e:
        pass

    # Create user if not yet there
    user = User.find_by_username(username=login)
    if not user:
        user = User(name=login, login=login, password='', email=email,
                    roles=[], text='LDAP user', email_verified=email_verified)
        try:
            user = user.create()
        except Exception as e:
            ApiError(str(e), 500)

    groups = [x.decode('utf-8').split(',')[0].split('=')[1] for x in ldap_result[0][1].get('memberOf', [])]

    # Check user is active
    if user.status != 'active':
        raise ApiError('User {} not active'.format(login), 403)
    user.update_last_login()

    scopes = Permission.lookup(login=login, roles=user.roles + groups)
    customers = get_customers(login=login, groups=[user.domain] + groups)

    auth_audit_trail.send(current_app._get_current_object(), event='basic-ldap-login', message='user login via LDAP',
                          user=login, customers=customers, scopes=scopes, resource_id=user.id, type='user',
                          request=request)

    # Generate token
    token = create_token(user_id=user.id, name=user.name, login=user.email, provider='ldap', customers=customers,
                         scopes=scopes, roles=user.roles, email=user.email, email_verified=user.email_verified)
    return jsonify(token=token.tokenize)
