# IAM infra that allows Jenkins to push artifacts to S3

resource "aws_iam_policy" "alerta_jenkins_codedeploy_iam_policy" {
  name = "alerta-jenkins-codedeploy-allow-s3"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:*"
      ],
      "Effect": "Allow",
      "Resource": ["${aws_s3_bucket.docker_alerta_codesource_bucket.arn}", "${aws_s3_bucket.docker_alerta_codesource_bucket.arn}/*"]
    },
    {
      "Action": [
        "ec2:Describe*"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Action": "sts:AssumeRole",
      "Resource": "arn:aws:iam::043714768218:role/jenkins-prod-*",
      "Effect": "Allow"
    }
  ]
}
EOF
}


data "aws_iam_policy_document" "allow_s3_to_jenkins_account" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "AWS"
      identifiers = ["arn:aws:iam::043714768218:root"]
    }
  }
}


resource "aws_iam_role" "alerta_jenkins_codedeploy_iam_role" {
  name = "alerta-jenkins-codedeploy-role"

  assume_role_policy = data.aws_iam_policy_document.allow_s3_to_jenkins_account.json
  tags               = local.common_tags
}

resource "aws_iam_policy_attachment" "alerta_jenkins_codedeploy_attach" {
  name       = "alerta-jenkins-codedeploy-allow-s3-attachment"
  roles      = ["${aws_iam_role.alerta_jenkins_codedeploy_iam_role.name}"]
  policy_arn = aws_iam_policy.alerta_jenkins_codedeploy_iam_policy.arn
}

resource "aws_iam_group" "jenkins_group" {
  name = "docker-alerta-jenkins"
}


# IAM stuff for codebuild/deploy/pipeline.

resource "aws_iam_role" "alerta_codebuild_role" {
  name = "codebuild-alerta-service-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "codebuild.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# This policy allows codebuild to access all its necessary resources to build and upload the docker image to ECR
# This includes CW logs permissions, ECR login/upload permissions, EC2 networking privileges so the container
# that builds the docker image can access our internal artifactory (as the accounts this is expected to run in are DX connected),
# as well as the S3 bucket and keys where our GHE will upload the code/config to kick off the process
resource "aws_iam_policy" "alerta_codebuild_policy" {
  name   = "codebuild-alerta-service-policy"
  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Resource": [
        "*"
      ],
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ]
    },
    {
      "Sid": "ECRPermissions",
      "Effect": "Allow",
      "Action": [
        "ecr:BatchCheckLayerAvailability",
        "ecr:CompleteLayerUpload",
        "ecr:GetAuthorizationToken",
        "ecr:InitiateLayerUpload",
        "ecr:PutImage",
        "ecr:UploadLayerPart"
      ],
      "Resource": [
        "*"
      ]
      },
    {
      "Effect": "Allow",
      "Action": [
        "ec2:CreateNetworkInterface",
        "ec2:DescribeDhcpOptions",
        "ec2:DescribeNetworkInterfaces",
        "ec2:DeleteNetworkInterface",
        "ec2:DescribeSubnets",
        "ec2:DescribeSecurityGroups",
        "ec2:DescribeVpcs"
      ],
      "Resource": "*"
    },
    {
      "Effect": "Allow",
      "Action": [
        "ec2:CreateNetworkInterfacePermission"
      ],
      "Resource": [
        "arn:aws:ec2:${var.region}:${data.aws_caller_identity.current.account_id}:network-interface/*"
      ],
      "Condition": {
        "StringEquals": {
          "ec2:Subnet": [${join(", ", formatlist("\"arn:aws:ec2:${var.region}:${data.aws_caller_identity.current.account_id}:subnet/%s\"", module.variables.private_subnet_ids))}],
          "ec2:AuthorizedService": "codebuild.amazonaws.com"
        }
      }
    },
    {
      "Effect": "Allow",
      "Action": [
        "s3:*"
      ],
      "Resource": [
        "${aws_s3_bucket.docker_alerta_codesource_bucket.arn}",
        "${aws_s3_bucket.docker_alerta_codesource_bucket.arn}/*",
        "${aws_s3_bucket.docker_alerta_codepipeline_bucket.arn}",
        "${aws_s3_bucket.docker_alerta_codepipeline_bucket.arn}/*"
      ]
    }
  ]
}
POLICY
}


resource "aws_iam_role_policy_attachment" "alerta_codebuild_policy_attachment" {
  role       = aws_iam_role.alerta_codebuild_role.name
  policy_arn = aws_iam_policy.alerta_codebuild_policy.id
}

# Give the pipeline access to poll the S3 bucket
resource "aws_iam_role_policy_attachment" "alerta_codebuild_managed_policy_attachment" {
  policy_arn = "arn:aws:iam::aws:policy/AWSCodeBuildAdminAccess"
  role       = aws_iam_role.alerta_codepipeline_role.name
}

# Simple codedeploy role for the deploy stage
resource "aws_iam_role" "alerta_codedeploy_role" {
  name = "alerta-codedeploy-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "codedeploy.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Use the default ECS codedeploy policy
resource "aws_iam_role_policy_attachment" "alerta_codedeploy_policy_attachment" {
  policy_arn = "arn:aws:iam::aws:policy/AWSCodeDeployRoleForECS"
  role       = aws_iam_role.alerta_codedeploy_role.name
}

# Simple codepipeline role
resource "aws_iam_role" "alerta_codepipeline_role" {
  name = "alerta-codepipeline-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "codepipeline.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Codepipeline Policy. This looks like a ridiculous perms set but it's basically what
# codepipeline generates automatically if you create a pipeline through the GUI
resource "aws_iam_policy" "alerta_codepipeline_allow" {
  name   = "codepipeline_alerta_allow"
  policy = <<POLICY
{
  "Statement": [
      {
          "Action": [
              "iam:PassRole"
          ],
          "Resource": "*",
          "Effect": "Allow",
          "Condition": {
              "StringEqualsIfExists": {
                  "iam:PassedToService": [
                      "cloudformation.amazonaws.com",
                      "elasticbeanstalk.amazonaws.com",
                      "ec2.amazonaws.com",
                      "ecs-tasks.amazonaws.com"
                  ]
              }
          }
      },
      {
          "Action": [
              "codecommit:CancelUploadArchive",
              "codecommit:GetBranch",
              "codecommit:GetCommit",
              "codecommit:GetUploadArchiveStatus",
              "codecommit:UploadArchive"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "codedeploy:CreateDeployment",
              "codedeploy:GetApplication",
              "codedeploy:GetApplicationRevision",
              "codedeploy:GetDeployment",
              "codedeploy:GetDeploymentConfig",
              "codedeploy:RegisterApplicationRevision"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "codestar-connections:UseConnection"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "elasticbeanstalk:*",
              "ec2:*",
              "elasticloadbalancing:*",
              "autoscaling:*",
              "cloudwatch:*",
              "s3:*",
              "sns:*",
              "cloudformation:*",
              "rds:*",
              "sqs:*",
              "ecs:*"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "lambda:InvokeFunction",
              "lambda:ListFunctions"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "opsworks:CreateDeployment",
              "opsworks:DescribeApps",
              "opsworks:DescribeCommands",
              "opsworks:DescribeDeployments",
              "opsworks:DescribeInstances",
              "opsworks:DescribeStacks",
              "opsworks:UpdateApp",
              "opsworks:UpdateStack"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "cloudformation:CreateStack",
              "cloudformation:DeleteStack",
              "cloudformation:DescribeStacks",
              "cloudformation:UpdateStack",
              "cloudformation:CreateChangeSet",
              "cloudformation:DeleteChangeSet",
              "cloudformation:DescribeChangeSet",
              "cloudformation:ExecuteChangeSet",
              "cloudformation:SetStackPolicy",
              "cloudformation:ValidateTemplate"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Action": [
              "codebuild:BatchGetBuilds",
              "codebuild:StartBuild"
          ],
          "Resource": "*",
          "Effect": "Allow"
      },
      {
          "Effect": "Allow",
          "Action": [
              "devicefarm:ListProjects",
              "devicefarm:ListDevicePools",
              "devicefarm:GetRun",
              "devicefarm:GetUpload",
              "devicefarm:CreateUpload",
              "devicefarm:ScheduleRun"
          ],
          "Resource": "*"
      },
      {
          "Effect": "Allow",
          "Action": [
              "servicecatalog:ListProvisioningArtifacts",
              "servicecatalog:CreateProvisioningArtifact",
              "servicecatalog:DescribeProvisioningArtifact",
              "servicecatalog:DeleteProvisioningArtifact",
              "servicecatalog:UpdateProduct"
          ],
          "Resource": "*"
      },
      {
          "Effect": "Allow",
          "Action": [
              "cloudformation:ValidateTemplate"
          ],
          "Resource": "*"
      },
      {
          "Effect": "Allow",
          "Action": [
              "ecr:DescribeImages"
          ],
          "Resource": "*"
      }
  ],
  "Version": "2012-10-17"
}
POLICY
}

# Give the pipeline access to poll the S3 bucket
resource "aws_iam_role_policy_attachment" "alerta_codepipeline_add_policy_attachment" {
  role       = aws_iam_role.alerta_codepipeline_role.name
  policy_arn = aws_iam_policy.alerta_codepipeline_allow.arn
}
