# the queue that will receive cross account SNS notifications
resource "aws_sqs_queue" "alerta_cloudwatch_queue" {
  name = "alerta-cross-account-queue"
  tags = local.common_tags
  message_retention_seconds = 86400 # one day
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.alerta_cloudwatch_deadletter.arn}\",\"maxReceiveCount\":3}"
}

# the deadletter queue is where failed to process messages (processed unsuccessfully > 3 times)
# get sent to for debugging
resource "aws_sqs_queue" "alerta_cloudwatch_deadletter" {
  name = "alerta-deadletter"
  message_retention_seconds = 604800 # seven days
  tags = local.common_tags
}

resource "aws_cloudwatch_metric_alarm" "deadletter_alarm" {
  alarm_name                = "Alerta SQS deadletter queue has message(s)"
  comparison_operator       = "GreaterThanOrEqualToThreshold"
  evaluation_periods        = "2"
  metric_name               = "NumberOfMessagesReceived"
  namespace                 = "AWS/SQS"
  statistic                 = "Sum"
  period                    = "60"
  threshold                 = "1"
  alarm_description         = "Monitors whether any malformed/unprocessable cloudwatch alarms pass through the alerta SQS queue"
  }

# Policy of accounts that are allowed to publish to this account
# Hacky way to have all allowed accounts be managed by one resource
# https://github.com/hashicorp/terraform/issues/12003#issuecomment-291228335
resource "aws_sqs_queue_policy" "alerta_cloudwatch_allowed_accs" {
  queue_url = aws_sqs_queue.alerta_cloudwatch_queue.id
  policy    = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "First",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.alerta_cloudwatch_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": ${jsonencode(var.account_topics)}
        }
      }
    }
  ]
}
POLICY
}

# iam role for lambda so it can integrate with sqs and push messages to sns
resource "aws_iam_role" "sqs_lambda_iam" {
  name               = "alerta-sqs-lambda-role"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
  {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
 ]
}
EOF
}

resource "aws_iam_policy" "sqs_lambda_policy" {
  name        = "alerta-lambda-allow-sqs-sns"
  description = "Allows the alerta forwarder lambda to pull messages from the central sqs queue and publish the messages into the master sns topic"
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
         "sqs:ReceiveMessage",
         "sqs:DeleteMessage",
         "sqs:GetQueueAttributes"
       ],
      "Resource": "${aws_sqs_queue.alerta_cloudwatch_queue.arn}"
    },
    {
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "sqs_lambda_managed_policy_attachment" {
  role       = aws_iam_role.sqs_lambda_iam.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole" # managed role
}

resource "aws_iam_role_policy_attachment" "sqs_lambda_policy_attachment" {
  role       = aws_iam_role.sqs_lambda_iam.name
  policy_arn = aws_iam_policy.sqs_lambda_policy.arn
}


# The lambda that forwards sqs messages to alerta
resource "aws_lambda_function" "sqs_forwarder_lambda" {
  filename      = "${path.module}/lambda/dist/alerta_cloudwatch.zip"
  function_name = "alerta_cross_account_sqs_forwarder"
  role          = aws_iam_role.sqs_lambda_iam.arn
  handler       = "cw_to_alerta.lambda_handler"

  source_code_hash = filebase64sha256("${path.module}/lambda/dist/alerta_cloudwatch.zip")

  runtime = "python3.7"
  
  timeout = 60
  
  vpc_config {
    subnet_ids = module.variables.private_subnet_ids
    security_group_ids = ["sg-5733632e"]
  }

}

resource "aws_lambda_event_source_mapping" "sqs_source_mapping" {
  event_source_arn = aws_sqs_queue.alerta_cloudwatch_queue.arn
  function_name    = aws_lambda_function.sqs_forwarder_lambda.arn
}