resource "aws_lb" "alerta_trivea_nlb" {
  name               = "alerta-trivea-nlb"
  internal           = true
  load_balancer_type = "network"
  subnets            = module.variables.private_subnet_ids
}

resource "aws_lb_target_group" "alerta_trivea_tg" {
  name        = "alerta-trivea-tg"
  port        = 443
  protocol    = "TLS"
  target_type = "ip"
  vpc_id      = module.variables.vpc_id
}

## We don't create any target_group_attachments since that data is not available
## to terraform. The accompanying lambda manages this instead.

resource "aws_lb_listener" "alerta_trivea_nlb_listener" {
  load_balancer_arn = "${aws_lb.alerta_trivea_nlb.arn}"
  port              = "443"
  protocol          = "TLS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = "arn:aws:acm:us-west-2:152548578290:certificate/3329fbd1-fb06-427a-a64d-57673ec581d0"

  default_action {
    type             = "forward"
    target_group_arn = "${aws_lb_target_group.alerta_trivea_tg.arn}"
  }
}

resource "aws_s3_bucket" "alerta_trivea_lambda_s3" {
  bucket = "alerta-trivea-lambda"
  acl    = "private"
}

resource "aws_lambda_function" "trivea_nlb_updater_lambda" {
  filename      = "${path.module}/lambda/dist/trivea_nlb_updater.zip"
  function_name = "alerta-trivea-nlb-updater"
  role          = aws_iam_role.trivea_role.arn
  handler       = "populate_NLB_TG_with_ALB.lambda_handler"

  source_code_hash = filebase64sha256("${path.module}/lambda/dist/trivea_nlb_updater.zip")

  runtime = "python3.7"

  timeout = 500

  environment {
    variables = {
      ALB_DNS_NAME = data.terraform_remote_state.alerta_infra.outputs.alerta_lb_dns_name,
      ALB_LISTENER = data.terraform_remote_state.alerta_infra.outputs.alerta_lb_listener_port,
      CW_METRIC_FLAG_IP_COUNT = true
      INVOCATIONS_BEFORE_DEREGISTRATION = 3,
      MAX_LOOKUP_PER_INVOCATION = 50,
      NLB_TG_ARN = aws_lb_target_group.alerta_trivea_tg.arn,
      S3_BUCKET = aws_s3_bucket.alerta_trivea_lambda_s3.id
    }
  }
}

resource "aws_cloudwatch_event_rule" "alerta_every_minute" {
  name        = "alerta-trivea-every-minute"
  description = "Fires once per minute."
  schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "alerta_trivea_event_target" {
  rule      = "${aws_cloudwatch_event_rule.alerta_every_minute.name}"
  arn       = "${aws_lambda_function.trivea_nlb_updater_lambda.arn}"
}

resource "aws_lambda_permission" "allow_cloudwatch" {
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.trivea_nlb_updater_lambda.function_name}"
  principal     = "events.amazonaws.com"
  source_arn    = "${aws_cloudwatch_event_rule.alerta_every_minute.arn}"
}

resource "aws_iam_role" "trivea_role" {
  name               = "video-alerta-trivea-${var.env}"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
  {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
 ]
}
EOF
}

resource "aws_iam_policy" "trivea_lambda_policy" {
  name        = "alerta-trivea-lambda"
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "LambdaLogging",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "S3",
      "Action": [
        "s3:Get*",
        "s3:PutObject",
        "s3:CreateBucket",
        "s3:ListBucket",
        "s3:ListAllMyBuckets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "ELB",
      "Action": [
        "elasticloadbalancing:Describe*",
        "elasticloadbalancing:RegisterTargets",
        "elasticloadbalancing:DeregisterTargets"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "CW",
      "Action": [
        "cloudwatch:putMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "alerta_trivea_lambda_policy_attachment" {
  role       = aws_iam_role.trivea_role.name
  policy_arn = aws_iam_policy.trivea_lambda_policy.arn
}

# Creates a DNS record for the service.
resource "aws_route53_record" "alerta_trivea_dns" {
  zone_id = "Z27SAAQG3G6TWS"
  name    = "video-alerta-trivea.${var.env}.vidops.twitch.a2z.com"
  type    = "A"

  alias {
    name                   = "${aws_lb.alerta_trivea_nlb.dns_name}"
    zone_id                = "${aws_lb.alerta_trivea_nlb.zone_id}"
    evaluate_target_health = true
  }
}

resource "aws_vpc_endpoint_service" "alerta_trivea_vpcec" {
  acceptance_required        = false
  network_load_balancer_arns = ["${aws_lb.alerta_trivea_nlb.arn}"]
  allowed_principals = ["arn:aws:iam::141456490005:root", "arn:aws:iam::330301539798:root"]
}
