import logging
import boto3
import sys
import json
import requests
import datetime
from requests.auth import AuthBase
import uuid

# region and env can be parameterized for multi-region deployment
ALERTA_ENDPOINT = 'https://video-alerta.us-west-2.staging.alerta.live-video.a2z.com/api/alert'
ALERTA_KEY      = 'KubpGl7yGIVMSuz9PFoaiZvoN-c7wFuK5tS_pak4'

logger = logging.getLogger()
logging.basicConfig(stream=sys.stdout, level=logging.INFO)

sqs = boto3.client('sqs')

STATE_TO_SEVERITY_MAP = {
    'ALARM': 'critical',
    'INSUFFICIENT_DATA': 'warning',
    'OK': 'ok'
}

# CustomJsonEncoder: Totally not stolen from Sparky
class CustomJsonEncoder(json.JSONEncoder):
    # some encoding stuff
    def default(self, o):
        if isinstance(o, (datetime.date, datetime.datetime)):
            return o.replace(microsecond=0).strftime('%Y-%m-%dT%H:%M:%S') + '.%03dZ' % (o.microsecond // 1000)
        elif isinstance(o, datetime.timedelta):
            return int(o.total_seconds())
        else:
            return json.JSONEncoder.default(self, o)

# ApiKeyAuth: Totally not more things stolen from Sparky
class ApiKeyAuth(AuthBase):

    def __init__(self, api_key=None, auth_token=None):
        self.api_key = api_key
        self.auth_token = auth_token

    def __call__(self, r):
        r.headers['Authorization'] = 'Key {}'.format(self.api_key)
        return r

# Provide sane defaults for CW alarms but enrich with TwitchTelemetry or 
# VidCS system metrics dimensions if applicable
def gather_dimensions(message):
    defaults = {'resource':  message['Trigger']['MetricName'],
                'environment': 'Staging',
                'service': [message['Trigger']['Namespace']],
                'event': message['AlarmName'],
                'tags': [message['AWSAccountId']]
                }
    # Form a searchable dict out of the list so lookups are less cumbersome
    dm = {item['name']: item['value'] for item in message['Trigger']['Dimensions']}

    if 'environment' in dm:
        defaults['environment'] = dm['environment'].capitalize()
    # service and role are analogous
    if 'service' in dm:
        defaults['service'][0] = dm['service']
    elif 'role' in dm:
        defaults['service'][0] = dm['role']
    # machine and host are analogous
    if 'machine' in dm:
        defaults['resource'] = dm['machine']
    elif 'host' in dm:
        defaults['resource'] = dm['host']
    return defaults


def form_alert(topic_arn, message):
    """Create message to go into queue"""
    dims = gather_dimensions(message)
    alert_body = {
        'resource': dims['resource'],
        'event': dims['event'],
        'environment': dims['environment'],
        'severity': STATE_TO_SEVERITY_MAP.get(message['NewStateValue'], 'unknown'),
        'correlate': [],
        'service': dims['service'],
        'group': dims['service'][0] + '-' + dims['environment'],
        'value': message['NewStateValue'],
        'text': "{}\n\nAlarmDescription: {}".format(message['NewStateReason'], message['AlarmDescription']),
        'tags': dims['tags'],
        'origin': topic_arn,
        'type': 'Cloudwatch',
        'createTime': datetime.datetime.now()
    }
    print(alert_body)
    return alert_body


def forward_to_alerta(message):
    try:
        raw_message = json.loads(message['body'])
        unloaded_alarm = json.loads(raw_message['Message'])
    except json.JSONDecodeError as e:
        logger.error("ERR: Failed to load message json: %s", e)
        return False
    try:
        formed_alarm = form_alert(raw_message['TopicArn'], unloaded_alarm)
    except Exception as e:
        logger.error("ERR: Failed to form message from cloudwatch alarm: %s", e)
        return False
    try:
        session = requests.Session()
        session.headers.update({
            'X-Request-ID': str(uuid.uuid4()),
            'Content-type': 'application/json'
        })
        hs = {'Authorization': "Key {}".format(ALERTA_KEY)}
        resp = session.post(ALERTA_ENDPOINT, data=json.dumps(formed_alarm, cls=CustomJsonEncoder), auth=ApiKeyAuth(ALERTA_KEY), timeout=30)
        print(resp.json())
    except Exception as e:
        logger.error("ERR: Failed to send alert to alerta: %s", e)
        return False
    return True


def lambda_handler(event, context):
    messages_processed = 0
    # Go through all messages at best effort
    for message in event['Records']:
        if forward_to_alerta(message):
            # "Alerts are cheap" - Alerta. we can dumbly forward the message
            # at best effort and delete them from the queue to keep the queue small
            # and let alerts be sent again
            messages_processed += 1


    logger.info("INFO: Successfully processed %s of %s messages", messages_processed, len(event['Records']))