class artm (
  $ensure              = $artm::params::ensure,
  $deploy_env          = $artm::params::deploy_env,
  $canary              = $artm::params::canary,
  $destfile            = $artm::params::destfile,
  $historydir          = $artm::params::historydir,
  $base_dir            = $artm::params::base_dir,
  $varnish_name        = $artm::params::varnish_name,
  $repo                = $artm::params::repo,
  $statsd_host         = $artm::params::statsd_host,
  $statsd_port         = $artm::params::statsd_port,
  $update_interval     = $artm::params::update_interval,
  $node_ttl            = $artm::params::node_ttl,
  $local_node          = $artm::params::local_node,
  $consul_host         = $artm::params::consul_host,
  $reptree_environment = $artm::params::reptree_environment,
) inherits artm::params {

  validate_string($destfile, $historydir, $base_dir, $varnish_name, $repo, $statsd_host)
  validate_integer($statsd_port)

  if $update_interval != undef {
    validate_re($update_interval, '^\d+(\.\d+)?(ns|us|ms|s|m|h)$', '$artm::update_interval is not a valid duration')
  }
  if ($node_ttl != undef) and ($node_ttl != '0') {
    validate_re($node_ttl, '^\d+(\.\d+)?(ns|us|ms|s|m|h)$', '$artm::node_ttl is not a valid duration')
  }

  # Place ARTM emergency rollback script.
  $rollback_script = 'rollback.sh'
  $rollback_path = "${base_dir}/artm/etc/${rollback_script}"

  file { $rollback_script:
    ensure  => $ensure,
    path    => $rollback_path,
    owner   => 'root',
    mode    => '0744',
    content => template('artm/rollback.sh.erb'),
  }

  # Set up ARTM.
  $artm_args = join_keys_to_values(delete_undef_values({
    '--statsd-server'   => "${statsd_host}:${statsd_port}",
    '--dest-file'       => $destfile,
    '--history-dir'     => $historydir,
    '--varnish-name'    => $varnish_name,
    '--update-interval' => $update_interval,
    '--node-ttl'        => $node_ttl,
    '--local-node'      => $local_node,
    '--consul-server'   => $consul_host,
    '--deploy-env'      => $deploy_env,
    '--reptree-env'     => $reptree_environment,
  }), '=')

  $artm_cmd = join([ './artm', join($artm_args, ' '), ], ' ')

  twitch::service { 'artm':
    ensure           => $ensure,
    env              => $deploy_env,
    canary           => $canary,
    predaemon        => [],
    base_dir         => $base_dir,
    daemon_cmd       => $artm_cmd,
    user             => 'root',
    service_repo     => $repo,
    manage_service   => true,
    monit_ensure     => $ensure,
    monit_check_port => 8778,
    monit_check_mode => 'http_cx',
    monit_down_kill  => true,
    subscribe        => Package['varnish']
  }

  # Place ARTM history cleanup script and set it up in cron.
  $cleanup_script = 'cleanup-history.sh'
  $cleanup_path = "${base_dir}/artm/etc/${cleanup_script}"

  file { $cleanup_script:
    ensure  => $ensure,
    path    => $cleanup_path,
    owner   => 'root',
    mode    => '0744',
    content => template('artm/history-cleanup.sh.erb'),
  }

  twitch_cron { 'artm-history-cleanup':
    ensure  => $ensure,
    user    => 'root',
    weekday => 'Monday',
    command => "${cleanup_path} 2>&1 | logger -p local2.info -t artm-history-cleanup",
    require => File[$cleanup_script],
  }

}
