#
# consul_service_list_lookup.rb
#

$LOAD_PATH.unshift File.dirname(__FILE__)

require 'libconsul'

module Puppet::Parser::Functions
  newfunction(:consul_service_list_lookup, :type => :rvalue, :doc => <<-EOS
Lookup a list of services from consul, return a list of items.
Useful for returning a list of IPs or a list of host names.
*Examples:*

consul_service_list_lookup("service_name", {"tag" => "staging"}, "ip")

Would return: [ "1.1.1.1", "1.1.1.2", "1.1.1.3" ]

consul_service_list_lookup("service_name", {"tag" => "staging"}, "node")

Would return: [ "foo1", "foo2", "foo3" ]

consul_service_list_lookup("service_name", {"tag" => "staging", dc => "sfo01"}, "fqdn")

Would return: [ "foo1.sfo01.justin.tv", "foo2.sfo01.justin.tv", "foo3.sfo01.justin.tv" ]

All optional arguments must be passed in as an hash.

Optional Arguments:

    tag: Filter service to to those including the tag
    consul_api_endpoint: Api server and port to query.
    passing: if true will only show services that are passing health checks. Defaults to false.
    dc: Datacenter to lookup service in. Defaults to the value of $pop. Specify "all" which queries each datacenter known.

Global Settings:

    consul_api_endpoint: Api server and port to query. The optional argument has higher priority.
                         Defaults to 'localhost:8500'.
    EOS
  ) do |args|
    raise(Puppet::ParseError, "consul_service_list_lookup(): Wrong number of arguments " +
      "given (#{args.size} for 2)") if args.size < 2

    service_name = args[0]
    opt          = {"dc" => lookupvar("pop"), "passing" => false}.merge(args[1] || {})
    list_item    = args[2]

    # Optional argument has highest priority, than the global value, than the default
    consul_server = opt["consul_api_endpoint"] || lookupvar("consul_api_endpoint") || "consul.internal.justin.tv"

    # Configure query args:
    query = []

    if opt["passing"]
      query << "passing"
    end

    if opt["tag"]
      query << "tag=#{opt["tag"]}"
    end

    result = []
    datacenters(consul_server, opt["dc"]).each do |dc|
      # Configure the URL
      url = URI::join("http://#{consul_server}/v1/health/service/", service_name)
      url.query = (query + ["dc=#{dc}"]).join("&")
      begin
        items = consul_api_call(url)
      # If the DC doesn't return valid info, skip it.
      rescue Puppet::ParseError
         next
      end
      if items != 'null'
        items.each do |item|
          service_addr = item["Service"]["Address"] || ""
          addr = item["Node"]["Address"]
          fqdn = item["Node"]["Node"]
          # Only lookup fqdn if it was requested (or all data requested).
          if (list_item == '' || list_item == 'fqdn') && !/^([a-z\d\.-]+)\.(justin|twitch)\.tv$/.match(fqdn)
            # Do a lookup on each node to get the fqdn tag from the nodeinfo service.
            node_services_url = URI::join("http://#{consul_server}/v1/catalog/node/", fqdn)
            node_services_url.query = "dc=#{dc}"
            node_services = consul_api_call(node_services_url)["Services"]
            if !node_services.nil? && node_services.key?("nodeinfo")
              fqdn_tag = node_services["nodeinfo"]["Tags"].select { |tag| tag.start_with?("fqdn=") }[0]
              fqdn = fqdn_tag.split("=")[1]
            else
              fqdn = "#{fqdn}.#{dc}.justin.tv"
            end
          end
          svc = {
            "port"    => item["Service"]["Port"],
            "ip"      => (service_addr.empty? ? addr : service_addr),
            "tags"    => item["Service"]["Tags"],
            "node"    => item["Node"]["Node"],
            "fqdn"    => fqdn,
            "dc"      => dc,
            "tagkeys" => {},
          }
          svc["tags"].each do |tag|
            kv = tag.split('=', 2)
            if kv.count == 2
              svc["tagkeys"][kv[0]] = kv[1]
            end
          end
          result << svc
        end
      end

    end

    if list_item != ''
      return result.map {|service| service[list_item]}.sort
    end
    return result
  end
end
