#
# consul_service_lookup.rb
#

$LOAD_PATH.unshift File.dirname(__FILE__)

require 'libconsul'

module Puppet::Parser::Functions
  newfunction(:consul_service_lookup, :type => :rvalue, :doc => <<-EOS
Lookup a list of services from consul

*Examples:*

    consul_service_lookup("service_name", {"tag" => "staging"})

Would return: [ {
                  node: "bar15",
                  dc: "sfo05",
                  ip: "1.1.1.1",
                  port: 80,
                  tags: [ "staging", "foo=bar" ],
                  tagkeys: { "foo": "bar" },
                },
                ...
              ]

All optional arguments must be passed in as an hash.

Optional Arguments:

    tag: Filter service to to those including the tag
    consul_api_endpoint: Api server and port to query.
    passing: if true will only show services that are passing health checks. Defaults to false.
    dc: Datacenter to lookup service in. Defaults to the value of $pop. Specify "all" which queries each datacenter known.

Global Settings:

    consul_api_endpoint: Api server and port to query. The optional argument has higher priority.
                         Defaults to 'localhost:8500'.
    EOS
  ) do |args|
    raise(Puppet::ParseError, "consul_service_lookup(): Wrong number of arguments " +
      "given (#{args.size} for 1)") if args.size < 1

    service_name = args[0]
    opt          = {"dc" => lookupvar('pop'), "passing" => false}.merge(args[1] || {})

    # Optional argument has highest priority, than the global value, than the default
    consul_server = opt["consul_api_endpoint"] || lookupvar("consul_api_endpoint") || "consul.internal.justin.tv"

    # Configure query args:
    query = []

    if opt["passing"]
      query << "passing"
    end

    if opt["tag"]
      query << "tag=#{opt["tag"]}"
    end

    result = []
    datacenters(consul_server, opt["dc"]).each do |dc|
      # Configure the URL
      url = URI::join("http://#{consul_server}/v1/health/service/", service_name)
      url.query = (query + ["dc=#{dc}"]).join("&")

      begin
        items = consul_api_call(url)
      # If the DC doesn't return valid info, skip it.
      rescue Puppet::ParseError
         next
      end

      if items != 'null'
        items.each do |item|
          service_addr = item["Service"]["Address"] || ""
          addr = item["Node"]["Address"]
          svc = {
            "port"    => item["Service"]["Port"],
            "ip"      => (service_addr.empty? ? addr : service_addr),
            "tags"    => item["Service"]["Tags"],
            "node"    => item["Node"]["Node"],
            "dc"      => dc,
            "tagkeys" => {},
          }
          svc["tags"].each do |tag|
            kv = tag.split('=', 2)
            if kv.count == 2
              svc["tagkeys"][kv[0]] = kv[1]
            end
          end
          result << svc
        end
      end

    end

    return result
  end
end
