class consul::config inherits consul {
  if $is_server {
    $metrics_prefix = $server_metrics_prefix
  } else {
    $metrics_prefix = $client_metrics_prefix
  }

  if $ui_enabled {
    $ui = true
  }

  if $is_server {
    # Bootstrap expect needs to be the quroum size (N/2)+1:
    # https://www.consul.io/docs/internals/consensus.html#deployment-table
    $bootstrap_expect = (pick($cluster_size,size($join)) + 0) / 2 + 1

    $reconnect_timeout = $consul::params::reconnect_timeout

    # Remove joining self wan cluster
    $join_wan_config = delete($join_wan, $consul::join_wan_ignore)

    # Validate that $cluster_size is an int:
    validate_re("$bootstrap_expect", '^\d+$')
    #make raft_multiplier a int
    $raft_multiplier = $raft_multiplier + 0
    $syslog_facility = 'local4'

    # FIXME/TODO VIDCS-3192: Maintain server current defaults
    # Perform proper rollout after global company break
    $limits = undef
  } else {
    # Non-server nodes don't need/want these config option.
    $join_wan_config = undef
    $bootstrap_expect = undef
    $raft_multiplier = undef

    $reconnect_timeout = undef

    # DTA-2542: For now, disable statsd dir on all clients to avoid an issue
    # overwhelming the statsd cluster_size
    $statsd_addr = undef

    # No limits for local agents
    if versioncmp($::consul::deb_version, '1.9.0') >= 0 {
      $limits = {
        'limits' => {
          'http_max_conns_per_client' => -1,
        }
      }
    }
    ####################
  }

  # Account for config changes/deprecations in newer versions
  if versioncmp($::consul::deb_version, '1.9.0') >= 0 {
    $_config_changes = delete_undef_values({
      # VIDCS-947 was missed after consul upgrade
      # Gate local script checks together with consul upgrade to 1.9.x
      'enable_local_script_checks' => true,

      # Former ui config has been deprecated in favor of the following:
      'ui_config' => delete_undef_values({
        'enabled' => $ui,
      })
    })
  }
  else {
    $_config_changes = {
      'enable_script_checks' => true,
      'ui'                   => $ui,
    }
  }

  $config = delete_undef_values(merge({
    'bind_addr'            => $bind,
    'advertise_addr'       => $advertise,
    'bootstrap'            => $bootstrap,
    'bootstrap_expect'     => $bootstrap_expect,
    'data_dir'             => $data_dir,
    'datacenter'           => $datacenter,
    'disable_remote_exec'  => $disable_remote_exec,
    'leave_on_terminate'   => $leave_on_exit,
    'disable_update_check' => $disable_update_check,
    'enable_syslog'        => $syslog,
    'node_name'            => $node_name,
    'server'               => $is_server,
    'reconnect_timeout'    => $reconnect_timeout,
    'retry_join'           => $join,
    'retry_join_wan'       => $join_wan_config,
    'syslog_facility'      => $syslog_facility,
    'log_level'            => $log_level,
    'ports'                => delete_undef_values({
      'serf_lan'           => $serf_lan,
    }),
    'telemetry'            => delete_undef_values({
      'statsd_address'     => $statsd_addr,
      'statsite_prefix'    => $statsite_prefix,
      'metrics_prefix'     => $metrics_prefix
    }),
    'performance'          => delete_undef_values({
      'raft_multiplier'    => $raft_multiplier
    }),
  }, $_config_changes, $limits))

  file { "${consul::config_dir}/main.json":
      mode    => '0644',
      owner   => 'root',
      group   => 'root',
      content => pretty_json($config),
      notify  => Service['consul'],
  }

  file { '/usr/local/bin/consul-execstartpre.sh':
    mode   => '0755',
    owner  => 'root',
    group  => 'root',
    source => 'puppet:///modules/consul/consul-execstartpre.sh',
  }

  twitch_systemd::unit_file { 'consul.service':
    content => template("${module_name}/systemd-consul.service.erb"),
    require => File['/usr/local/bin/consul-execstartpre.sh'],
  }
}
