define consul::service (
  $ensure             = 'present',
  $service_name       = $name,
  $port               = 0,
  $consul_tags        = [],
  $check_script       = undef,
  $check_http         = undef,
  $check_tcp          = undef,
  $check_interval     = undef,
  $maintmode_kv_watch = false,
  $global_tags        = hiera_array('consul::global_tags', []),
  $service_ip         = undef,
  $meta               = undef,
) {
  # In order to support validating both strings and ints we need to wrap $port
  # in quotes.
  validate_re("$port", '^\d+$')
  validate_array($consul_tags)
  validate_bool($maintmode_kv_watch)

  include ::consul

  if $check_script != undef {
    $args = shell_split($check_script)
  } else {
    $script = $check_script
  }

  # Starting in some unknown version of consul (we know it applies to > 0.9.0), there is a change made to the way that consul handles scripts with certain fields missing.
  # e.g. the config format changes, and delete_undef_values isn't sufficient to address the change.
  if $check_interval == undef {
    if $check_script != undef {
      notify { "WARN: \$check_script is set and \$check_interval is NOT set for consul::service ${name}": }
    }
    $service_check = undef
  } else {
    $service_check   = delete_undef_values({
      'script'   => $script,
      'args'     => $args,
      'interval' => $check_interval,
      'http'     => $check_http,
      'tcp'      => $check_tcp,
    })
  }

  $config = {
    'service' => delete_undef_values({
      'id'      => $name,
      'name'    => $service_name,
      'tags'    => delete_undef_values(flatten([$consul_tags, $global_tags])),
      'address' => $service_ip,
      # We can convert $port (possibly a string) to a port by adding 0
      'port'    => 0 + $port,
      'check'   => $service_check,
      'meta'    => $meta,
    })
  }

  file { "${consul::config_dir}/service_${name}.json":
    ensure  => $ensure,
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
    content => pretty_json($config),
    notify  => Class['::consul::reload_service'],
  }

  if $maintmode_kv_watch {
    $maintmode_kw_watch_ensure = $ensure
  } else {
    $maintmode_kw_watch_ensure = 'absent'
  }

  consul::watch { "maintmode-by-kv_${name}":
    ensure  => $maintmode_kw_watch_ensure,
    type    => 'key',
    handler => "${consul::maintmode_kv_watch_handler} '${name}'",
    key     => "service/${service_name}/enabled/${consul::node_name}",
    require => File[$consul::maintmode_kv_watch_handler]
  }
}
