define consul::watch (
  $type,
  $handler,
  $ensure      = 'present',
  $token       = undef,
  $datacenter  = undef,
  $key         = undef,
  $keyprefix   = undef,
  $service     = undef,
  $service_tag = undef,
  $passingonly = undef,
  $state       = undef,
  $event_name  = undef,
) {
  include ::consul

  validate_string($type)
  validate_string($handler)

  # optional common params
  validate_string($token)
  validate_string($datacenter)

  # gather common parameters
  $common = {
    type       => $type,
    handler    => $handler,
    token      => $token,
    datacenter => $datacenter,
  }

  # gather type-specific parameters
  case $type {
    'key': {
      if !$key { fail('Must specify the key to watch.') }
      validate_string($key)
      $specific = { key => $key }
    }
    'keyprefix': {
      if !$keyprefix { fail('Must specify the keyprefix to watch') }
      validate_string($keyprefix)
      $specific = { keyprefix => $keyprefix }
    }
    'services': {
      $specific = { }
    }
    'nodes': {
      $specific = { }
    }
    'service': {
      if !$service { fail('Must specify the service to watch') }
      validate_string($service)
      validate_string($service_tag)
      validate_bool($passingonly)
      $specific = {
        service     => $service,
        tag         => $service_tag,
        passingonly => $passingonly,
      }
    }
    'checks': {
      if !$service and !$state { fail('Must specify either a service (to subscribe to its all of its checks) or a state (to subscribe to all checks in that state)') }
      if $service and $state { fail ('Cannot specify both a service and a state; pick one') }
      validate_string($service)
      validate_string($state)
      $specific = {
        service => $service,
        state   => $state,
      }
    }
    'event': {
      validate_string($event_name)
      $specific = { name => $event_name }
    }
    default: {
      fail("Invalid watch type (got '${type}')")
    }
  }

  # merge common and specific configs
  $config = {
    watches => [ delete_undef_values(merge($common, $specific)) ]
  }

  file { "${consul::config_dir}/watch_${title}.json":
    ensure  => $ensure,
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
    content => pretty_json($config),
    notify  => Class['::consul::reload_service'],
  }
}
