class consul_template (
  $ensure          = $consul_template::params::ensure,
  $consul_addr     = $consul_template::params::consul_addr,
  $wait            = $consul_template::params::wait,
  $max_stale       = $consul_template::params::max_stale,
  $retry           = $consul_template::params::retry,
  $backoff         = $consul_template::params::backoff,
  $max_backoff     = $consul_template::params::max_backoff,
  $version         = $consul_template::params::version,
  $consul_loglevel = $consul_template::params::consul_loglevel,
  $package_name    = $consul_template::params::package_name,
  $dedup_enabled   = $consul_template::params::dedup_enabled,
) inherits consul_template::params {
  require consul

  validate_string($consul_addr)
  validate_re($ensure, '^(present|absent)$', "consul_template::ensure can only be 'absent' or 'present'. \$version changes the version")

  if versioncmp($version, '0.19.0') >= 0 {
    $legacy_consul = false
  } else {
    $legacy_consul = true
  }

  case $ensure {
    'present': {
      $pkg_ensure = $version
    }
    'absent': {
      $pkg_ensure = 'absent'

      # Collect any consul_template service and define it absent, global overrides all else
      Consul_template::Service <| |> {
        ensure => $ensure,
      }
    }
  }

  package { $package_name:
    ensure  => $pkg_ensure,
  }

  file { '/usr/local/bin/consul-template-test':
    ensure => $ensure,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
    source => "puppet:///modules/${module_name}/consul-template-test.sh",
  }

  concat { $config_file:
    ensure => $ensure,
    notify => Daemontools::Supervise['consul-template'],
  }

  if $ensure == 'present' {
    concat::fragment { 'consul_template_00_preamble':
      target  => $config_file,
      content => template("${module_name}/preamble.hcl.erb"),
      before  => Daemontools::Supervise['consul-template'],
    }
  }

  daemontools::supervise { 'consul-template':
    ensure         => $ensure,
    daemon         => "consul-template -config ${config_file}",
    daemon_dir     => '/var/lib/service',
    options        => '2>&1',
    wd             => '/tmp',
    syslog         => 'local3',
    restart_signal => 'HUP',
  }

}
