#
define consul_template::service(
  $source,
  $destination,
  $command,
  $ensure          = 'present',
  $backup          = false,
  $auto_gen        = false,
  $command_timeout = false,
) {
  include consul_template

  validate_absolute_path($source)
  validate_absolute_path($destination)
  validate_bool($backup, $auto_gen)
  validate_re($ensure, [ '^present$', '^absent$' ])

  if $ensure == 'present' {
    concat::fragment { "consul_template_${name}":
      target  => $consul_template::config_file,
      content => template("${module_name}/single_service.hcl.erb"),
      before  => Daemontools::Supervise['consul-template'],
    }
  }

  if $auto_gen and $ensure == 'present' {
    # puppet needs a way to make sure the file is generated on disk in a
    # resource that can be ordered against
    # XXX: This could be made into a native type/provider
    # we use unless to run consul-template in dry run mode, which only
    # outputs to stdout if template will render changes as compared to
    # the configured file on disk, meaning: run consul-template
    # once to generate file only if dry mode produces output (exit >0)
    $logging  = '-syslog -syslog-facility=local3 -log-level=INFO'
    $template = "-template '${source}:${destination}:${command}'"
    $once_cmd = "consul-template ${template} ${logging} -once"
    exec { "Generate ${destination} once":
      command => $once_cmd,
      onlyif  => "test -f ${source}",
      unless  => "consul-template-test ${logging} ${template}",
      require => [
        Class['consul_template'],
        File[$source],
      ],
    }
  }

  if defined(Concat[$source]) {
    Concat[$source] ~> Daemontools::Supervise['consul-template']
  } else {
    File[$source] ~> Daemontools::Supervise['consul-template']
  }
}
