#!/bin/bash

# disable non-relevant shellchecks
# shellcheck disable=SC1090,SC1091,SC2154

# stdout and stderr are redirected to syslog
exec &> >(logger -p local3.info -t sns-updater)

# how long do we try to push changes in case of failures
RETRY_TIMEOUT=3600

# we are interested in changes on this interface only
INTERFACE="team0"

# array of sns topics to which we push
TOPIC_ARNS=(
    "arn:aws:sns:us-west-2:190359952526:hls-dns-crud-updater"
    "arn:aws:sns:us-east-2:625436545767:hls-dns-crud-updater"
)

# directory to track pids of the launched hooks instances. We need this because it
# can happen so that another hook is started while previous one is not competed yet.
PID_DIR="/run/sns-updater"

# let's ensure directory exists
mkdir -p "$PID_DIR"
# pid file for this process is created
touch "$PID_DIR/$BASHPID"

# in this loop we check that no other hook istances are running
# any older hook instances are terminated
for pid_file in "$PID_DIR"/*; do
    pid=$(basename "$pid_file")

    # we skip our own pid
    ((pid == BASHPID)) && continue

    # if we found pid for the non running process we remove the pid file
    kill -0 "$pid" 2>/dev/null || {
        rm -f "$pid_file"
        continue
    }

    # processes with older pids are killed
    [ "/proc/$BASHPID" -nt "/proc/$pid" ] && kill "$pid"
done

# if there are saved settings from the previous run let's source them
[ -f "/run/${INTERFACE}.save" ] && . "/run/${INTERFACE}.save"

# if this is on private IP, need proxy to talk to AWS
[[ -f "/etc/profile.d/proxy.sh" ]] && . /etc/profile.d/proxy.sh

# Wait for millliner to be ready before proceeding
until systemctl is-active --quiet video-milliner; do
    echo "Milliner not ready yet, waiting..."
    sleep 10
done

# this script can be called either from dhclient or from puppet. dhclient will set environment
# variables including "reason". If "reason" is not set this means script is started from the puppet.
source="dhclient"
[ -z "$reason" ] && source="puppet" && reason="puppet"

case $source in
    dhclient)
        echo "Triggered via dhclient"

        # we are interested only for events on the INTERFACE
        [ "$interface" != "$INTERFACE" ] && exit
        ;;
    puppet)
        echo "Triggered via puppet or manually"

        # let's get ip from the interface
        new_ip_address="$(ip -o addr | grep -oP '^\d+:\steam0\s+inet \K[\d.]*')"

        old_ip_address=$saved_ip_address
        ;;
esac

# Make sure we actually have an IP address before continuing
# Different stages (e.g. PREINIT) of dhclient hooks might not have data
if [[ -z "$new_ip_address" ]]; then
    echo "Skipping. No IP address found"
    exit
fi

# read host MCK
. <(facter -p twitch_environment twitch_role pop|sed -e 's/ => /=/')

# let's compile json message for sns
json=$(cat <<JSON
{
    "source": "$source",
    "reason": "$reason",
    "hostname": "$(hostname -f)",
    "timestamp": $(date +%s),
    "pop": "$pop",
    "twitch_role": "$twitch_role",
    "twitch_environment": "$twitch_environment",
    "old_ip_address": "$old_ip_address",
    "new_ip_address": "$new_ip_address"
}
JSON
)

# source milliner environment to get credentials
. /etc/milliner-client-env.sh

# track for which regions we successfully pushed the data
succeeded_regions=""
now=$(date +%s)
success=0

export AWS_STS_REGIONAL_ENDPOINTS=regional
# in case of failures we will try for RETRY_TIMEOUT
while (( $(date +%s) - now < RETRY_TIMEOUT)); do
    # loop through sns topic arns
    for topic_arn in "${TOPIC_ARNS[@]}"; do

        # topic arn contains region, let's extract it
        region=$(echo "$topic_arn" | cut -f4 -d:)

        # have pushed to this region already? If yes let's skip it
        [[ "$succeeded_regions=" =~ $region ]] && continue

        # topic arn contains account number, let's get it
        account_number=$(echo "$topic_arn" | cut -f5 -d:)
        # we need to assume role and set variables with credentials

        read -r AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SECURITY_TOKEN < <(
            aws --region "$region" sts assume-role \
            --role-arn "arn:aws:iam::${account_number}:role/hls-crud-sns-publish-role" \
            --role-session-name hls-crud-sns-publish-role-session-name \
            --query '[Credentials.AccessKeyId,Credentials.SecretAccessKey,Credentials.SessionToken]' \
            --output text)
        export AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SECURITY_TOKEN

        # now we can try to push data to sns
        echo "Publishing to ${topic_arn}"
        aws --region "$region" sns publish --topic-arn "$topic_arn" --message "$json" --output text && {
            # in case of success we increment counter
            ((success += 1))
            # and note the region
            succeeded_regions+=$region
        }

        # Reset this else we will continue operating under the assumed role
        unset AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SECURITY_TOKEN
    done
    # if we are done in both regions - stop the loop
    ((success == 2)) && echo "Published to SNS!" && break
    # wait and retry
    sleep 10
done

# let's save ip settings so we could use it on the next invocation
cat <<EOF >"/run/${INTERFACE}.save"
saved_ip_address=$new_ip_address
EOF
