# Class to manage /etc/resolv.conf
class core::dns inherits core {
  validate_string($core::searchdomain)
  validate_array($core::nameservers)

  # Install local resolver
  if str2bool($core::use_local_resolver) {
    if versioncmp($::lsbmajdistrelease, '18.04') >= 0 {
      notify { 'Bionic unsupported':
        message => 'Bionic is currently not supported by the local unbound resolver'
      }
    }
    else {
      class {'twitch_unbound':
        role                => 'forwarder',
        val_permissive_mode => true,
        }
      $subscribes = str2bool($facts['dhcp_enabled']) ? {
        true    => Class['core::dhcp'],
        default => undef,
      }
      # just replace all current name servers with localhost,
      exec { 'ensure localhost resolver':
        command     => "sed -i -e 's/^nameserver.*$/nameserver 127.0.0.1/g' /etc/resolv.conf",
        unless      => "grep -qE '^nameserver 127.0.0.1$' /etc/resolv.conf",
        onlyif      => "test -f /etc/unbound/unbound.conf",
        refreshonly => true,
        subscribe   => $subscribes,
        require     => Class['twitch_unbound'],
      }
    }
  }

  # if dhcp is enabled on the box (99.9% of cases)
  if str2bool($facts['dhcp_enabled']) {
    include core::dhcp
  } else {
    # if dhcp is not enabled on this host
    # puppet should manage the content of resolv.conf
    file{ '/etc/resolv.conf':
      ensure  => 'file',
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/resolv.conf.erb"),
    }
  }

}
