# Filebeat is our log shipping agent, which sends local logs to be indexed in a
# central location.
class core::filebeat (
  $logstash_hosts      = [],
  $standard_output_tag = 'syslog_elasticsearch',
  $ensure              = 'absent',
  $manage_repo         = false,
) {

  $_base_log_tags = delete_undef_values([
    $::twitch_role ? {
      undef   => undef,
      default => "role_${::twitch_role}",
    },
    $twitch_environment ? {
      undef   => undef,
      default => "environment_${twitch_environment}",
    },
    $::twitch_environment ? {
      undef   => undef,
      default => "environment_${::twitch_environment}",
    },
    $::twitch_team ? {
      undef   => undef,
      default => "team_${::twitch_team}",
    },
    $::pop ? {
      undef => undef,
      default => "pop_${::pop}",
    },
    $cluster ? {
      undef   => undef,
      default => "cluster_${cluster}",
    },
    $::rack_name ? {
      undef   => undef,
      default => "rack_${::rack_name}",
    },

    # Mark dirty logs so they're easy to find if we're cleaning things
    str2bool($::clean) ? {
      true    => undef,
      default => 'dirty',
    },
  ])

  # Replace dashes with underscores in tags so they get indexed as a whole
  # instead of by component.
  $_log_tags = regsubst($_base_log_tags, '-', '_', 'G')

  if $ensure == 'absent' {
    $directory_ensure = 'absent'
    $package_ensure   = 'absent'
    $service_ensure   = 'absent'
    $service_enable   = false
  } else {
    $directory_ensure = 'directory'
    $package_ensure   = '5.6.4'
    $service_ensure   = 'running'
    $service_enable   = true
  }

  file { '/var/lib/filebeat':
    ensure  => $directory_ensure,
    force   => true,
    recurse => true,
    owner   => 'root',
    group   => 'root',
    mode    => '0750',
  }

  # ensure absent, leave defn here for now to remove
  class { '::filebeat':
    package_ensure => $package_ensure,
    service_ensure => $service_ensure,
    service_enable => $service_enable,
    manage_repo    => $manage_repo,
    logging        => {
      level     => 'warning',
      to_syslog => false,
      to_files  => true,
      files     => {
        path      => '/var/log/filebeat',
        name      => 'filebeat.log',
        keepfiles => 7,
      },
    },
    shipper        => {
      # Name defaults to the hostname, which isn't fully qualified.
      'name' => $::ipaddress_primary,
    },
    beat_name      => $::ipaddress_primary,
    outputs        => {
      'logstash' => {
        'hosts'       => sort($logstash_hosts),
        'loadbalance' => true,
      },
    },
    tags           => $_log_tags,
  }


  # Restarts Filebeat daily.  Put into place to workaround bug where Filebeat
  # holds log files open after they've been deleted.
  cron::daily { 'filebeat':
    ensure  => $ensure,
    command => '/usr/sbin/service filebeat restart > /dev/null 2>&1',
    user    => 'root',
    hour    => fqdn_rand(24,'filebeat_hour_seed'),
    minute  => fqdn_rand(60, 'filebeat_minute_seed'),
  }
}
