# == Define: courier::dirs
#
# === Parameters
#
# [*basedir*]
#  Base directory for installation, Default: /opt/twitch
#
# [*user*]
#  User to own directories, Default: jtv
#
# [*group*]
#  Group to own directories, Default: jtv
#
# [*codedeploy_compat*]
#  Simply create 'current' symlink in appdir for compatability with staged-deploy.
#
# === Example
# # This example will create the following directories:
# #  /opt/twitch/myservice
# #  /opt/twitch/myservice/etc
# #  /opt/twitch/myservice/releases
# #  /opt/twitch/myservice/manifests
# courier::dirs{ 'myservice':}
#
define courier::dirs (
  $ensure            = present,
  $basedir           = hiera('twitch_basedir','/opt/twitch'),
  $user              = 'jtv',
  $group             = 'jtv',
  $codedeploy_compat = false,
) {

  $appdir       = "${basedir}/${name}"
  $confdir      = "${appdir}/etc"
  $releasesdir  = "${appdir}/releases"
  $shareddir    = "${appdir}/shared"
  $manifestsdir = "${appdir}/manifests"

  validate_absolute_path($confdir)
  validate_absolute_path($releasesdir)
  validate_absolute_path($shareddir)
  validate_absolute_path($manifestsdir)

  $ensure_dir = $ensure ? {
    'absent' => absent,
    default  => directory,
  }

  $ensure_link = $ensure ? {
    'absent' => absent,
    default  => link,
  }

  file { $appdir:
    ensure => $ensure_dir,
    force  => true,
    backup => false,
    owner  => $user,
    group  => $group,
    mode   => '0775',
  }

  if $codedeploy_compat {
    # only manage symlink for staged-deploy compatability.
    file { "${appdir}/current":
      ensure => $ensure_link,
      force  => true,
      target => "/opt/codedeploy/${name}/current",
    }
  } else {
    file { $confdir:
      ensure  => $ensure_dir,
      backup  => true,
      owner   => $user,
      group   => $group,
      mode    => '0775',
      require => File[$appdir],
    }
  
    file { $releasesdir:
      ensure  => $ensure_dir,
      backup  => false,
      owner   => $user,
      group   => $group,
      mode    => '0775',
      require => File[$appdir],
    }
  
    file { $shareddir:
      ensure  => $ensure_dir,
      backup  => true,
      owner   => $user,
      group   => $group,
      mode    => '0775',
      require => File[$appdir],
    }
  
    file { $manifestsdir:
      ensure  => $ensure_dir,
      backup  => false,
      owner   => $user,
      group   => $group,
      mode    => '0775',
      require => File[$appdir],
    }
  }
}
