# == Class: courier
#
# Installs courier for Twitch deployments
#
# [*version*]
#  Version of courier to install, Default: latest
#
# [*conf_file*]
#  Path to courier config file, Default: /etc/courier.conf
#
# [*s3_bucket*]
#  s3 bucket name that holds the build artifacts, Default: <from hiera>
#
# [*conf_owner*]
#  User with ownership of the config file, Default: root
#
# [*conf_group*]
#  Group with ownership of the config file, Default: root
#
# [*conf_mode*]
#  Permissions on the config file, Default 0444 (string)
# [*access_key*]
#  S3 Access ID for the deployment artifact bucket, Default: <from hiera>
#
# [*secret_key*]
#  S3 Secret Key for the deployment artifact bucket, Default: <from hiera>
#
# [*build_retention*]
#  Default number of builds to keep, Default: <from hiera> Fallback: 5
#
# [*proxy*]
#  If machine needs a proxy, add to config, Default: <depends on $::ipaddress_primary>
#
# Config example:
# {
#   "s3_bucket": "build-bucket",
#   "access_key": "ASDFJKLQWERTYHJKL",
#   "secret_key": "YOUCANTHAVEMYSECRET",
#   "build_retention": "3",
#   "http_proxy": "proxy.blah"
# }
#
class courier (
  $version            = $courier::params::version,
  $conf_file          = $courier::params::conf_file,
  $s3_bucket          = $courier::params::s3_bucket,
  $conf_owner         = $courier::params::conf_owner,
  $conf_group         = $courier::params::conf_group,
  $conf_mode          = $courier::params::conf_mode,
  $access_key         = $courier::params::access_key,
  $secret_key         = $courier::params::secret_key,
  $build_retention    = $courier::params::build_retention,
  $proxy              = $courier::params::proxy,
  $max_s3_connections = $courier::params::max_s3_connections,
  $consul_fallback_dc = $courier::params::consul_fallback_dc
) inherits courier::params {

  validate_absolute_path($conf_file)
  validate_string($s3_bucket,$conf_owner,$conf_group,$conf_mode,$access_key,$secret_key)

  # Make my config
  $config = delete_undef_values({
    's3_bucket'         => $s3_bucket,
    'access_key'        => $access_key,
    'secret_key'        => $secret_key,
    'build_retention'   => $build_retention,
    'http_proxy'        => $proxy,
    'maxs3connections'  => $max_s3_connections,
  })

  validate_hash($config)

  # Lets not have random configs with creds on boxes that don't need them
  $conf_ensure = $version ? {
    'absent' => 'absent',
    default  => present,
  }

  package { 'courier':
    ensure => $version
  }

  file{ $conf_file:
    ensure  => $conf_ensure,
    owner   => $conf_owner,
    group   => $conf_group,
    mode    => $conf_mode,
    content => pretty_json($config),
    require => Package['courier'],
  }
}
