# == Define: courier::install
#
# Installs Twitch software using courier
#
# === Parameters:
# [*env*]
#  Environment used as part of consul query to get current release for that environment, Default: *required*
#
# [*repo*]
#  Github repo project name, Default: *required*
#
# [*ensure*]
#  Create or remove resources.
#
# [*basedir*]
#  Twitch base directory, Default: <hiera:twitch_basedir>
#
# [*type*]
#  Type of install, Options: tar, docker, Default: tar
#
# [*image*]
#  Required when type => docker, name of image, Default: undef
#
# [*user*]
#  User to run courier and own directories, Default: jtv
#
# [*group*]
#  Group to own directories, Default: jtv
#
# [*consul_host*]
#  Consul host to connect to, Default: localhost:8500
#
# [*extra_flags*]
#  Extra flags to pass to courier command.  Must be a string and begin with --, Default: ''
#  Example: '--option1 value1 --option2 value2'
#
# [*enable_courierd*]
#  Enable courierd monitoring of shasums for service, Default: false
#
# [*codedeploy_compat*]
#  Simply create 'current' symlink in appdir for compatability with staged-deploy.
#
# === Example
# This example will install owl in the default locations with the version listed for production
# Courier will extract owl artifact to /opt/twitch/owl/releases/<sha> and move the symlink
#  to point to this revision
#
# courier::install{ 'owl':
#   env  => 'production',
#   repo => 'web/owl',
# }
#
# Example courier command:
#  courier tar install --consul-host localhost:8500 --repo 'web/owl' --environment production --dir /opt/twitch/owl
define courier::install (
  $env,
  $repo,
  $ensure            = present,
  $basedir           = hiera('twitch_basedir','/opt/twitch'),
  $type              = 'tar',
  $image             = undef,
  $user              = 'jtv',
  $group             = 'jtv',
  $consul_host       = hiera('consul_host', 'localhost:8500'),
  $extra_flags       = '',
  $enable_courierd   = false,
  $restart           = false,
  $tmpdir            = hiera('courier::install::tmpdir', '/tmp/'),
  $codedeploy_compat = false,
) {
  if ! $codedeploy_compat {
    include courier
    include courierd::config
    include consul
  }

  validate_bool($restart)

  $appdir = "${basedir}/${name}"

  if ($type == 'docker') and ($image == undef) {
    fail('FAIL: courier::install type => docker requires image to be set')
  }

  if ($extra_flags != '') and ($extra_flags !~ /^--/) {
    fail('FAIL: courier::install extra_flags does not begin with --')
  }

  if ($type != 'docker') and ($type != 'tar') {
    fail('Not a valid type.  Valid types: docker, tar')
  }

  courierd::path{ $name:
    path   => $appdir,
    enable => $enable_courierd,
  }

  $type_flags = $type ? {
    'docker' => "--image ${image}",
    'tar'    => "--dir ${appdir}",
    default  => '',
  }

  $courier_install_command = "courier ${type} install --consul-fallback-dc ${::courier::consul_fallback_dc} --consul-host ${consul_host} --repo ${repo} --environment ${env} ${type_flags} ${extra_flags}"
  $courier_restart_command = "courier ${type} restart --consul-fallback-dc ${::courier::consul_fallback_dc} --consul-host ${consul_host} --repo ${repo} --environment ${env} ${type_flags} ${extra_flags}"
  $courier_status_command  = "courier ${type} status --consul-fallback-dc ${::courier::consul_fallback_dc} --consul-host ${consul_host} --repo ${repo} --environment ${env} ${type_flags} ${extra_flags}"

  courier::dirs{ $name:
    ensure            => $ensure,
    basedir           => $basedir,
    user              => $user,
    group             => $group,
    codedeploy_compat => $codedeploy_compat,
  }

  if !$::disable_consul and $ensure == 'present' and !$codedeploy_compat {
    exec{ "courier install ${name}":
      command     => $courier_install_command,
      path        => ['/bin', '/sbin', '/usr/bin', '/usr/sbin', '/usr/local/bin', '/usr/local/sbin'],
      environment => ["TMPDIR=${tmpdir}"],
      user        => $user,
      timeout     => 600,
      unless      => $courier_status_command,
      require     => [
        # I know you're really excited to fix the courier race condition problem
        # you found, but unfortunately it's not going to work :(
        # We've tried the same change you're about to make 3-4 times before and
        # it's always been reverted pretty much immediately, because it causes
        # puppet dependency cycles. :( :( :( :( :( :( :(
        #
        # For luls:
        # https://git-aws.internal.justin.tv/systems/puppet/pull/2548
        # https://git-aws.internal.justin.tv/systems/puppet/pull/2923
        #
        # https://git-aws.internal.justin.tv/systems/puppet/pull/2013
        # https://git-aws.internal.justin.tv/systems/puppet/pull/2016
        #
        # https://git-aws.internal.justin.tv/systems/puppet/commit/6ac3f24c77bc9a556a05ee43c9b8c3a94dc1d8bc
        # https://git-aws.internal.justin.tv/systems/puppet/commit/361c9dc9ae0040178370d23282b5eb823ad5f796
        Class['courier'],
        Courier::Dirs[$name],
        Class['twitch_useraccess::sudoers'],
      ],
    }

    # consul service is not defined if $::disable_consul is true
    if defined(Service['consul']) {
      Exec <| title == "courier install ${name}" |> {
        require +> Service['consul'],
      }
    }
  }

  if ($restart and !$codedeploy_compat) {
    exec{ "courier restart ${name}":
      command => $courier_restart_command,
      path    => ['/bin', '/sbin', '/usr/bin', '/usr/sbin', '/usr/local/bin', '/usr/local/sbin'],
      user    => $user,
      timeout => 600,
      require => [
        Exec["courier install ${name}"]
      ],
    }
  }
}
