# == Class: courierd
#
# Installs courierd for managing Twitch code on servers
#
# [*version*]
#  Version of courierd to install, Default: latest
#
# [*user*]
#  User to run service, Default: nobody
#
# [*pub_key*]
#  Path to the courierd config file
#
# [*fail_service_key*]
#  Pagerduty service key that will receive alerts.
#
# [*warn_service_key*]
#  Pagerduty service key that will receive warnings.

class courierd (
  $version          = $courierd::params::version,
  $user             = $courierd::params::user,
  $pub_key          = $courierd::params::pub_key,
  $fail_service_key = $courierd::params::fail_service_key,
  $warn_service_key = $courierd::params::warn_service_key,
  $cgroup_rules     = $courierd::params::cgroup_rules,
) inherits courierd::params {

  include courierd::config

  $ensure_resource = $version ? {
    'absent' => absent,
    default  => present,
  }

  package { 'courierd':
    ensure => $version,
    notify => Service['courierd'],
  }

  file{ $pub_key:
    ensure  => $ensure_resource,
    owner   => 'root',
    group   => 'root',
    mode    => '0444',
    source  => 'puppet:///modules/courierd/courierd_key.pub', #Pub key
    require => Package['courierd'],
  }

  #lets validate the hash for our cgroup rules
  validate_hash($cgroup_rules)
  daemontools::supervise { "courierd":
    ensure         => $ensure_resource,
    manage_service => true,
    daemon_dir     => '/var/lib/service',
    user           => $user,
    syslog         => local3,
    enable_cgroups => true,
    cgroups_rules   => $cgroup_rules,
    wd             => '/',
    daemon         => "courierd -fail-service-key=${fail_service_key} -warn-service-key=${warn_service_key} 2>&1",
    predaemon      => [
      "export GOMAXPROCS=4"
    ],
    require        => [
      File[$pub_key],
      Class['courierd::config'],
    ],
  }
}
