# == Define: supervise
#
# Create a service that is monitored by daemontools.
#
# === Parameters
#
# [*wd*]
#   Working directory to run the command in.
#
# [*daemon*]
#   Binary to run. This is run via exec so it should be be a single command
#   containing no pipes or parameters.
#
# [*options*]
#   Options and parameters to pass to the daemon.
#
# [*syslog*]
#   Configures daemontools to log to syslog.
#
#   Valid options:
#     local2  = Logs to /var/log/jtv.log
#     local3  = Logs to /var/log/jtv/<tagname>.log
#     remote  = Logs to central syslog
#     cwlogs  = Logs to cloudwatch using twitch_svclogs::logger
#
# [*predaemon*]
#   Shell commands to run prior to the daemon. Usually for exporting shell parameters.
#
# [*user*]
#   Before 'exec'ing to the daemon the user will be changed to the defined value.
#
# [*ensure*]
#   Whether the daemontools supervise entry should exist. This assumes bringing down
#   a running service.
#
# [*down*]
#   Valid values:
#   true  => ensure a down file exists for the service
#   force => ensure a down file exists for the service and make sure the service is down in supervise
#   false => remove the down file if it exists
#   undef => don't change anything
#
# [*tagname*]
#   Set the tagname to log to syslog under. By default this is set to $name.
#
# [*fdlimit*]
#   Change the file descriptor limit set by ulimit. By default this is 262144.
#
# [*daemon_dir*]
#   The directory where daemon files exist (contrary to where daemontools looks for them)
#
# [*manage_service*]
#   Should we control restarting the service after update of run scripts? Default: false
#
# [*physical_cd*]
#   Forces cd to move to the actual location of any symlinks. By default set to false.
#
# [*logger_datetime*]
#  Optional format string for twitch_svclog::logger ($syslog => 'cwlogs')
#
# [*enable_cgroups*]
#   Bool that will call the cgroups define to create cgroup for this service,
#     default to false
#   Valid values:
#    true  => calls cgroups define to create cgroup directories for this service
#    false => will not call cgroup define but will not remove if they exist
#
# [*cgroups_basedir*]
#   Base directory of cgroups.  Not likely to change
#    Default: /sys/fs/cgroup
# [*cgroups_rules*]
#  Accepts a hash of hashes
#   Default: {}
#   Ex: { 
#         'memory' => {
#           'memory.swappiness'     => '0',
#           'memory.limit_in_bytes' => '9223372036854775807',
#          },
#          'blkio' => {
#            'blkio.weight' => '1000',
#          },
#        }
#
# [*cgroups*]
#   Array of cgroups to enable if enable_cgroups = true
#     Default: ['cpuacct','memory']
#
# [*cgroups_namespace*]
#   Sets the namespace under which the service will live in the cgroups
#     Default: service
#     Results in /sys/fs/cgroup/<cgroup>/$cgroups_namespace/$name
#    
# === Examples
#
#   daemontools::supervise { 'echo_server':
#     daemon  => "/usr/bin/nc",
#     options => "-l 5000",
#     user    => "nobody",
#     wd      => "/tmp/"
#   }
#
define daemontools::supervise (
  $wd,
  $daemon,
  $env               = pick($::twitch_environment, $twitch_environment, 'production'),
  $concurrency       = 0,
  $physical_cd       = false,
  $options           = '',
  $syslog            = '',
  $predaemon         = [],
  $user              = '',
  $need_aux_groups   = undef,
  $ensure            = 'present',
  $down              = undef,
  $tagname           = undef,
  $fdlimit           = 262144,
  $daemon_dir        = '/etc/service',
  $manage_service    = false,
  $restart_signal    = 'TERM',
  $logger_datetime   = undef,
  $enable_cgroups    = false,
  $cgroups_basedir   = '/sys/fs/cgroup',
  $cgroups_rules     = {},
  $cgroups_namespace = 'service',
  $cgroups           = [
    'cpuacct',
    'memory',
  ],
) {
  require daemontools
  validate_integer($concurrency)
  validate_bool($physical_cd, $manage_service, $enable_cgroups)
  validate_array($predaemon,$cgroups)
  validate_absolute_path($daemon_dir,$cgroups_basedir)
  validate_hash($cgroups_rules)
  validate_re($restart_signal, '^(TERM|STOP|CONT|HUP|ALRM|INT)$')

  if $tagname {
    $tag = $tagname
  } else {
    $tag = $name
  }

  # setuidgid drops auxiliary groups by defult
  # use sudo -u if they are required for perms
  if $need_aux_groups and $user != '' {
    $runstring = 'sudo -u'
  } else {
    $runstring = 'setuidgid'
  }

  $runas = $user ? {
    /^\w+/ => "${runstring} ${user} ",
    default => '',
  }

  # lets make sure that we're using a dir that is on the whitelist
  if !($daemon_dir in $daemontools::params::daemon_dirs) {
    $join_dirs = join($daemontools::params::daemon_dirs, ', ')
    fail("${daemon_dir} is not one of '${join_dirs}'")
  }

  $service_dir  = "${daemon_dir}/${name}"
  $service_run  = "${service_dir}/run"
  $service_down = "${service_dir}/down"

  $cgroups_rules_cmd = "/usr/local/bin/cgroups_rules_${name}.sh"
  $cgroups_paths     = prefix($cgroups,"${cgroups_basedir}/")
  $cgroups_paths_svc = suffix($cgroups_paths,"/${cgroups_namespace}/${name}")
  $cgroups_procs_svc = suffix($cgroups_paths_svc,'/cgroup.procs')
  $cgroups_add       = prefix($cgroups_procs_svc,'/bin/echo $$ > ')
  $cgroups_dirs_cmd  = prefix($cgroups_paths_svc,'mkdir -p ')

  if ($enable_cgroups) and ($::lsbmajdistrelease >= '12') {
    $real_enable_cgroups = true
  } else {
    $real_enable_cgroups = false
  }

  if ($real_enable_cgroups) {
    # Make sure all $cgroups are available before we go further
    $allcgroups    = split($::available_cgroups,',')
    $valid_cgroups = member($allcgroups,$cgroups)

    if ($valid_cgroups) {
      cgroups::service{ $name:
        cgroups   => $cgroups,
        basedir   => $cgroups_basedir,
        rules     => $cgroups_rules,
        namespace => $cgroups_namespace,
      }
    } else {
      $string_cgroups = join($cgroups,',')
      notify{ "${name}: cgroup unavailable":
        loglevel => err,
        message  => "cgroup unavailable for Daemontools::Supervise[${name}], available cgroups: ${::available_cgroups}, provided cgroups: ${string_cgroups}",
      }
    }
  }

  # Injecting cgroup related steps before other $predaemon settings
  # if cgroups are enabled and all elements in $cgroups are available
  # If neither of these are true, we remove cgroup related lines from the run file
  if ($real_enable_cgroups) and ($valid_cgroups) {
    $predaemon_full = flatten([$cgroups_dirs_cmd,$cgroups_rules_cmd,$cgroups_add,$predaemon])
  } else {
    $predaemon_full = $predaemon
  }

  # if concurrency > 0, run under 'consul lock' to enforce per-pop concurrency limits.
  # it will wait to run for up to a year before giving up and letting daemontools restart it.
  # monitor-retry 0 ensures the service will shut down immediately if it loses control of its lock-
  # most usually from inability to communicate with the cluster.
  if ($concurrency > 0) {
    $consul_lock = "consul lock -verbose -try=8760h -monitor-retry=5 -n=${concurrency} -name=${name} supervise/${env}/${name}"
  }

  if $ensure == false or $ensure =~ /^(false|absent)$/ {
    # if we're not supposed to exist
    $ensure_dir     = $ensure
    $force_dir      = true
    $ensure_down    = 'absent'
    $service_ensure = 'stopped'

    # forcefully remove service, only if we have an actual directory in either supported daemon_dirs
    exec { "remove $name":
      command => "/usr/local/bin/remove_service ${name}",
      onlyif  => "/usr/bin/test -d /var/lib/service/${name} || /usr/bin/test -d /etc/service/${name}",
      before  => File[$service_dir],
    }
  } else {
    # if we're supposed to exist, dirs are too
    $ensure_dir  = 'directory'

    # but if we want down
    if $down == true or "${down}" =~ /^(true|force)$/ {
      $ensure_down    = 'present'
      $service_ensure = 'stopped'
    } else {
      # all other cases we dont want down
      $ensure_down    = 'absent'
      $service_ensure = 'running'
    }
  }

  # if down is not specified, dont even define the file (so we dont remove it)
  if $down != undef {
    file { $service_down:
      ensure => $ensure_down,
      owner  => 'root',
      group  => 'root',
      mode   => '0644',
      before => File[$service_run],
    }
  }

  file { $service_dir:
    ensure => $ensure_dir,
    force  => $force_dir,
  }

  file { $service_run:
    ensure  => $ensure,
    mode    => '0755',
    content => template("${module_name}/supervise_run.erb"),
  }

  # create log dirs/files if we need them
  if $syslog == 'cwlogs' {
    twitch_svclogs::logger { $name:
      env             => $env,
      runas_user      => $user,
      datetime_format => $logger_datetime,
    }

  } elsif $syslog {
    $log_dir = "${service_dir}/log"
    $log_run = "${log_dir}/run"

    file { $log_dir:
      ensure => $ensure_dir,
      force  => $force_dir,
    }

    file { $log_run:
      ensure  => $ensure,
      mode    => '0755',
      content => "#!/bin/bash\n\nexec logger -p ${syslog}.info -t ${tag}\n",
      before  => File[$service_run],
      notify  => Exec["refresh $log_dir"],
    }

    exec { "refresh $log_dir":
      command     => "/usr/bin/svc -h $log_dir",
      refreshonly => true,
    }
  }

  # if we're in a sane path, define a service
  if ($daemon_dir == '/var/lib/service') {
    # if we had logging we should make sure we wait for those run files
    $svc_require = $syslog ? {
      ''       => File[$service_run],
      'cwlogs' => Twitch_svclogs::Logger[$name],
      default  => File[$log_run, $service_run],
    }
    if $manage_service {
      Service[$name] {
        subscribe => File[$service_run],
      }
    }
    service { $name:
      ensure   => $service_ensure,
      require  => $svc_require,
      provider => 'twitch_daemontools',
      path     => $daemon_dir,
      restart  => $restart_signal,
    }
  } elsif $down == 'force' and $ensure == 'present' {
    # if no real service defined, but we want the files around
    # but we still want it down
    exec { "force kill $name":
      command => "/usr/bin/svc -dk $service_dir",
      onlyif  => "/usr/bin/test -d $real_dir || /usr/bin/test -L $real_dir",
      require => File[$service_run],
    }
  }
}
