# Darkseer is a network stats daemon for baremetal hosts
class darkseer (
  $ensure           = $darkseer::params::ensure,
  $env              = $darkseer::params::env,
  $http_port        = $darkseer::params::http_port,
  $node_name        = $darkseer::params::node_name,
  $hostclass        = $darkseer::params::hostclass,
  $statsd_host_port = $darkseer::params::statsd_host_port,
) inherits darkseer::params {
  validate_re($ensure, [ '^present$', '^absent$' ])
  validate_re($env, [ '^production$', '^staging$', '^canary$' ])
  validate_integer($http_port)

  $consul_tags = $env ? {
    'canary' => [ $env, 'canary' ],
    default  => [ $env ],
  }

  consul::service { 'darkseer':
    ensure      => $ensure,
    consul_tags => $consul_tags,
    port        => $http_port,
  }

  courier::install { 'darkseer':
    ensure            => $ensure,
    env               => $env,
    repo              => 'video/darkseer',
    basedir           => '/opt/twitch',
    before            => Service['darkseer'],
    codedeploy_compat => true
  }

  $options = prefix(delete_undef_values({
    'app'              => 'video.darkseer',
    'addr'             => ":${http_port}",
    'environment'      => $env,
    'node'             => $node_name,
    'statsd-host-port' => $statsd_host_port,
    'hostclass'        => $hostclass,
  }), '-')
  $daemon_options = join(join_keys_to_values($options, '='), ' ')

  twitch_systemd::unit_file {'darkseer.service':
    ensure  => $ensure,
    content => template('darkseer/darkseer.service.erb'),
  }

  $enabled = $ensure ? {
    'present' => true,
    default   => false,
  }

  # Install service and restart on unit file change.
  # Darkseer *CAN* be restarted during a puppet run
  service { 'darkseer':
    ensure    => $enabled,
    enable    => $enabled,
    subscribe => Twitch_systemd::Unit_file['darkseer.service'],
  }

  # Create a check to forward darkseer health status to nagios.
  twitch_servicecheck::passive { 'video-darkseer':
    command        => "envoy-check --host 127.0.0.1 --port ${http_port}",
    interval       => 1, # minutes between checks
    retry          => 2, # times to fail before reporting failure
    retry_interval => 1, # minutes between checks once reporting failure
  }
}
