# == Class: discovery
#
# Installs and manages the discovery service
#
# === Parameters
#
# [*discovery_basedir*]
#  Base directory to find discovery code, Default: /home/jtv/discovery_deploy
#
# [*discovery_environment*]
#  Parameter for daemontools and for consul registration, Default: production
#
# [*user*]
#  User to run the daemontools service, Default: jtv
#
# [*port*]
#  Port for discovery service, Default: 9090
#
# [*pg_master_host*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_slave_host*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_master_port*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_slave_port*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_name*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_user*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*pg_pass*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*raven_domain*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*raven_project*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*raven_api_user*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*raven_api_pass*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*jax_url*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*giantbomb_url*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*giantbomb_key*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*statsd*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*redis*]
#  Parameter for daemontools, Default: <see params.pp>
#
# [*gomaxprocs*]
#  Sets GOMAXPROCS for daemontools, Default: 8
class discovery (
  $discovery_basedir         = $discovery::params::discovery_basedir,
  $discovery_environment     = $discovery::params::env,
  $user                      = $discovery::params::user,
  $port                      = $discovery::params::port,
  $pg_master_host            = $discovery::params::pg_master_host,
  $pg_slave_host             = $discovery::params::pg_slave_host,
  $pg_master_port            = $discovery::params::pg_master_port,
  $pg_slave_port             = $discovery::params::pg_slave_port,
  $pg_name                   = $discovery::params::pg_name,
  $pg_user                   = $discovery::params::pg_user,
  $pg_pass                   = $discovery::params::pg_pass,
  $raven_domain              = $discovery::params::raven_domain,
  $raven_project             = $discovery::params::raven_project,
  $raven_api_user            = $discovery::params::raven_api_user,
  $raven_api_pass            = $discovery::params::raven_api_pass,
  $giantbomb_url             = $discovery::params::giantbomb_url,
  $giantbomb_key             = $discovery::params::giantbomb_key,
  $statsd                    = $discovery::params::statsd,
  $redis                     = $discovery::params::redis,
  $gomaxprocs                = $discovery::params::gomaxprocs,
) inherits discovery::params {

  include twitch::haproxy::backend
  include twitch_geoip
  include twitch::packages

  realize(Package['pkg-config'])

  if $discovery_environment == 'production-aws' or $discovery_environment == 'production' {
    include discovery::haproxy::backend_discovery_database_master_production
    include discovery::haproxy::backend_discovery_database_slave_production
  } else {
    include discovery::haproxy::backend_discovery_database_master_staging
    include discovery::haproxy::backend_discovery_database_slave_staging
    $jax_url = 'http://jax-internal-staging.us-west2.justin.tv'
    $searchindexer_stream_name = 'searchindexer-staging'
    $game_updates_topic_arn  = 'arn:aws:sns:us-west-2:465369119046:discovery_staging_game_updates'
    $gameCommunity_topic_arn = 'arn:aws:sns:us-west-2:465369119046:discovery_staging_game_addition_or_deletion'
  }

  $deploy_env = hiera('clean', 'false') ? {
    'true'  => "clean-${env}",
    default => $env,
  }

  $raven = "https://${raven_api_user}:${raven_api_pass}@${raven_domain}/${raven_project}"

  ::twitch_nginx::vhost{ 'godiscovery':
    content => template("${module_name}/nginx/${module_name}.conf.erb"),
  }

  ::daemontools::supervise { $title:
    daemon_dir => '/var/lib/service',
    syslog    => 'local3',
    wd        => $discovery_basedir,
    user      => $user,
    predaemon => [
      'exec 2>&1',
      "export STATSD_HOSTPORT='${statsd}'",
      "export GOMAXPROCS=${gomaxprocs}",

      "export ENVIRONMENT=${discovery_environment}",
      "export PORT=${port}",

      "export PG_MASTER_HOST=${pg_master_host}",
      "export PG_MASTER_PORT=${pg_master_port}",

      "export PG_SLAVE_HOST=${pg_slave_host}",
      "export PG_SLAVE_PORT=${pg_slave_port}",

      "export PG_DBNAME=${pg_name}",
      "export PG_USER=${pg_user}",
      "export PG_PASS=${pg_pass}",

      "export RAVEN_DSN=${raven}",
      "export REDIS_HOSTPORT=${redis}",
      "export JAX_BASE_URL=${jax_url}",

      "export GIANTBOMB_URL=${giantbomb_url}",
      "export GIANTBOMB_API_KEY=${giantbomb_key}",

      "export MEMCACHE_HOST=${memcache_host}",
      "export MEMCACHE_PORT=${memcache_port}",

      "export SEARCHINDEXER_STREAM_NAME=${searchindexer_stream_name}",
      "export GAME_UPDATES_TOPIC_ARN=${game_updates_topic_arn}",
      "export GAMECOMMUNITY_TOPIC_ARN=${gameCommunity_topic_arn}"
    ],
    daemon    => 'current/discovery',
    options   => "-addr :${port}",
  }

  file { '/etc/service/discovery_jaxstore/consul_lock_wrapper':
    ensure  => present,
    owner   => 'jtv',
    group   => 'jtv',
    mode    => '0755',
    source  => 'puppet:///modules/discovery/consul_lock_wrapper',
    require => Class['twitch::users::jtv'],
  }

  ::daemontools::supervise { "discovery_jaxstore":
    syslog    => 'local3',
    wd        => $discovery_basedir,
    user      => $user,
    predaemon => [
      'exec 2>&1',
      "export STATSD_HOSTPORT='${statsd}'",
      "export GOMAXPROCS=${gomaxprocs}",

      "export PG_MASTER_HOST=${pg_master_host}",
      "export PG_MASTER_PORT=${pg_master_port}",
      "export PG_DBNAME=${pg_name}",
      "export PG_USER=${pg_user}",
      "export PG_PASS=${pg_pass}",

      "export JAX_BASE_URL=${jax_url}",

      "export SHELL=/etc/service/discovery_jaxstore/consul_lock_wrapper",
    ],
    daemon    => "consul lock locks/discovery_jaxstore_${discovery_environment} current/discovery_jaxstore -environment ${discovery_environment}",
  }

  ::daemontools::supervise { "discovery_index":
    syslog    => 'local3',
    wd        => $discovery_basedir,
    user      => $user,
    predaemon => [
      'exec 2>&1',
      "export STATSD_HOSTPORT='${statsd}'",
      "export GOMAXPROCS=${gomaxprocs}",

      "export PG_MASTER_HOST=${pg_master_host}",
      "export PG_MASTER_PORT=${pg_master_port}",
      "export PG_DBNAME=${pg_name}",
      "export PG_USER=${pg_user}",
      "export PG_PASS=${pg_pass}",

      "export JAX_BASE_URL=${jax_url}",
      "export GIANTBOMB_URL=${giantbomb_url}",
      "export GIANTBOMB_API_KEY=${giantbomb_key}",
      "export SEARCHINDEXER_STREAM_NAME=${searchindexer_stream_name}",
      "export GAME_UPDATES_TOPIC_ARN=${game_updates_topic_arn}",
      "export GAMECOMMUNITY_TOPIC_ARN=${gameCommunity_topic_arn}",

      "export SHELL=/etc/service/discovery_jaxstore/consul_lock_wrapper",
    ],
    daemon    => "consul lock locks/discovery_index_${discovery_environment} current/index -environment=${discovery_environment}",
  }

  consul::service { 'discovery':
    port        => $port,
    consul_tags => [$deploy_env],
  }
}
