# Install wireguard packages for kernel module and tools
class dumbo_wireguard::packages (
  # Wireguard tools only work with the kernel module with the same version; take
  # care if using `latest` as it may break the tools until the module has been
  # reloaded
  $ensure     = $::dumbo_wireguard::params::ensure,
  $tools_only = $::dumbo_wireguard::params::tools_only,
) inherits ::dumbo_wireguard::params {
  validate_re($ensure, '^(installed|latest|present)$')
  validate_bool($tools_only)

  if $::osfamily == 'Debian' {
    include apt

    $osname = downcase($::operatingsystem)

    case $osname {
      'ubuntu' : {
        apt::source { 'wireguard' :
          location          => "http://ppa.launchpad.net/wireguard/wireguard/${osname}",
          release           => $facts['lsbdistcodename'],
          repos             => 'main',
          key               => {
            'id'     => 'E1B39B6EF6DDB96564797591AE33835F504A1A25',
            'server' => 'pgp.mit.edu',
          },
          include           => {
            'src' => false,
          },
          required_packages => [ 'wireguard-tools', 'wireguard-dkms' ],
        }
      }
      default : {
        apt::pin {'wireguard':
          packages => ['wireguard-dkms', 'wireguard-tools'],
          release  => 'experimental',
          priority => 501,
          require  => Apt::Source['debian_unstable'],
        }

      }
    }
  }

  unless $tools_only {
    package { 'wireguard-dkms':
      ensure => $ensure,
    }
  }

  package { 'wireguard-tools':
    ensure => $ensure,
  }

  $ensure_dir = $ensure ? {
    absent  => absent,
    default => directory,
  }

  $ensure_file = $ensure ? {
    absent  => absent,
    default => present,
  }

  file { '/etc/wireguard':
    ensure  => $ensure_dir,
    owner   => 'root',
    group   => 'root',
    mode    => '0700',
    purge   => true,
    recurse => true,
    require => Package['wireguard-tools'],
  }

  file { '/etc/systemd/system/wireguard@.service.d':
    ensure  => $ensure_dir,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    purge   => true,
    recurse => true,
  }

  file { '/etc/systemd/system/wireguard@.service':
    ensure => $ensure_file,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => 'puppet:///modules/dumbo_wireguard/wireguard@.service',
  }

}
