define gatekeeper::instance(
  $public_addr,
  $private_addr,
  $env,
  $sock_name,
  $sock_base_path = '/var/run/gatekeeper',
  $ensure = 'present',
  $backup = false,
) {
  validate_bool($gatekeeper::enable_edgeevent, $backup)
  validate_hash($gatekeeper::extra_flags)
  validate_re($ensure, [ '^present$', '^absent$' ])

  $statsd_addr = "${gatekeeper::statsd_host}:${gatekeeper::statsd_port}"

  $role = $backup ? {
    true  => 'backup',
    false => 'primary',
  }

  $sock_path = "${sock_base_path}/${sock_name}"

  $options = {
    'node'               => $gatekeeper::host_name, # deprecating soon
    'nname'              => $gatekeeper::node_name,
    'mode'               => $gatekeeper::mode,
    'public_addr'        => $public_addr,
    'private_addr'       => $private_addr,
    'sockpath'           => $sock_path,
    'edgeevent_enabled'  => $gatekeeper::enable_edgeevent,

    'statsd_group'       => $gatekeeper::stats_group,
    'statsd_host'        => $statsd_addr,
    'stream_ttl'         => $gatekeeper::stream_ttl,
    'role'               => $role,
  }

  $all_options = prefix(delete_undef_values(merge($options, $gatekeeper::extra_flags)), '-')

  ::daemontools::supervise { $name:
    ensure         => $ensure,
    syslog         => 'local3',
    wd             => "${gatekeeper::basedir}/gatekeeper",
    user           => 'nobody',
    manage_service => true,
    predaemon      => [
      "/bin/mkdir -p /var/run/edgeevent && chmod -R 777 /var/run/edgeevent",
      "/bin/mkdir -p ${sock_base_path} && chmod -R 777 ${sock_base_path}",
      "systemctl is-active --quiet video-milliner",
      "source /etc/milliner-client-env.sh",
      "export GOMAXPROCS=${gatekeeper::gomaxprocs}",
      "export HTTPS_PROXY=http://proxy.internal.justin.tv:9797/",
      "export https_proxy=http://proxy.internal.justin.tv:9797/",
      'exec 2>&1',
    ],
    daemon         => 'current/gatekeeper',
    options        => join(join_keys_to_values($all_options, '='), ' '),
  }

  $split_addr = split($public_addr, ':')
  $consul_port = $split_addr[1]
  $deployment_tags = $env ? {
    'canary' => ['production', $env],
    default  => any2array($env),
  }
  $consul_tags = $backup ? {
    true  => concat(['backup'], $deployment_tags),
    false => $deployment_tags,
  }

  ::consul::service { $name:
    ensure       => $ensure,
    service_name => 'gatekeeper',
    port         => $consul_port,
    consul_tags  => $consul_tags,
    require      => Daemontools::Supervise[$name],
  }
}
