# == Class: ingest_proxy
#
# Setup and configures the Ingest Proxy Service.  This service consists of:
#  - Anycast IP and Bird configured
#  - Anycast Enforcer configured
#  - Nginx vhost that sends metrics to graphite
#
# == Parameters
# [*anycast_ips*]
#  Hash with information regarding the interface and anycast IP address.  Type: Hash
#   ex. {
#         'lo:myservice' => {
#            'ipaddress' =>  '10.10.10.10'
#         }
#       }
#
# [*upstreams*]
#  Array of upstreams to be configured in Nginx.  Type: Array
#   ex. [backend-host-0.internal.justin.tv,backend-host-1.internal.justin.tv,backend-host-2.internal.justin.tv]
#
# [*stats_prefix*]
#  Prefix to the name of the graphite metric.  Type: String
#
# [*graphite_endpoint*]
#  Hostname of the graphite endpoint
#
# [*loopback_ensure*]
#  Ensure parameter of the bird::loopback define
#
# [*nginx_restart_command*]
#  Command used to restart nginx
#
# [*nagios_service_name*]
#  Name of service registered with nagios
#
# [*nrdp_folder*]
#  Location of NRDP folder
#
# [*enable_nginx*]
#  Setup nginx portion of the serice.  Type: Bool
#
# [*enable_ssl*]
#  Enable or disable ssl within nginx.  Type: Bool
#
# [*enable_anycast*]
#  Setup anycast configuration and interfaces.  Type: Bool
#
# [*enable_anycast_enforcer*]
#  Setup and configure anycast health enforcer.  Type: Bool
#
class ingest_proxy (
  $anycast_ips                     = $ingest_proxy::params::anycast_ips,
  $findingest_lb                   = $ingest_proxy::params::findingest_lb,
  $stats_prefix                    = $ingest_proxy::params::stats_prefix,
  $graphite_endpoint               = $ingest_proxy::params::graphite_endpoint,
  $loopback_ensure                 = $ingest_proxy::params::loopback_ensure,
  $nginx_restart_command           = $ingest_proxy::params::nginx_restart_command,
  $nagios_service_name             = $ingest_proxy::params::nagios_service_name,
  $nrdp_folder                     = $ingest_proxy::params::nrdp_folder,
  $enable_nginx                    = $ingest_proxy::params::enable_nginx,
  $enable_ssl                      = $ingest_proxy::params::enable_ssl,
  $enable_anycast                  = $ingest_proxy::params::enable_anycast,
  $enable_anycast_enforcer         = $ingest_proxy::params::enable_anycast_enforcer,
  $nameservers                     = hiera('twitch_nameservers'),
  $ingest_env                      = $ingest_proxy::params::ingest_env,
) inherits ingest_proxy::params {
  validate_hash($anycast_ips)
  validate_string($findingest_lb)
  validate_bool($enable_nginx)
  validate_bool($enable_anycast)
  validate_bool($enable_anycast_enforcer)
  validate_bool($enable_ssl)
  validate_string($ingest_env)

  # setup anycast health checks for ingest proxies
  $anycast_instances = keys($anycast_ips)
  $tags              = prefix($anycast_instances, "nagios_passive_check=${nagios_service_name} ")
  $more_tags         = suffix($tags, "|${nagios_service_name}")

  if $enable_anycast {
    include bird::multiservice

    create_resources(bird::loopback, $anycast_ips, {
      auto    => true,
      ensure  => $loopback_ensure,
    })

    if $enable_anycast_enforcer {
      twitch_anycast::health_enforcer::daemon { $anycast_instances:
        health_url     => 'http://127.0.0.1:80',
        nagios_service => $nagios_service_name,
        nrdp_folder    => $nrdp_folder,
        require        => Bird::Loopback[$anycast_instances],
      }
    }
  }

  if $enable_nginx {
    include twitch::crossdomain
    twitch_nginx::vhost { 'ingestproxy':
      content => template("${module_name}/nginx-ingestproxy.conf.erb"),
      require => Class['sysctl::values'],
    }

    twitch_nginx::vhost { 'graphite_config':
      content => "graphite_config server=${graphite_endpoint} prefix=${stats_prefix} frequency=10;\n",
    }

    consul::service { 'ingest-proxy':
      consul_tags => concat($more_tags, "fqdn=${::fqdn}"),
      port        => 80,
      require     => Twitch_nginx::Vhost['ingestproxy'],
    }

    if $enable_ssl {

      $wildcard_twitch_cert = join([
        "{{ key \"video/ssl_keys/${ingest_env}/wildcard_twitch_tv/cert\" }}",
        "{{ key \"video/ssl_keys/${ingest_env}/wildcard_twitch_tv/intermediate\" }}",
      ], "\n")

      twitch_ssl_certificate::certificate { 'wildcard_twitch_tv_cert':
        cert_identifier      => 'wildcard_twitch_tv_cert',
        cert_filesystem_path => '/etc/ssl/certs/wildcard_twitch_tv.pem',
        cert_mode            => '0644',
        cert_owner           => 'root',
        cert_group           => 'root',
        cert_contents        => $wildcard_twitch_cert,
        cert_restart_cmd     => $nginx_restart_command,
        key_identifier       => 'wildcard_twitch_tv_key',
        key_filesystem_path  => '/etc/ssl/private/wildcard_twitch_tv.key',
        key_mode             => '0640',
        key_owner            => 'root',
        key_group            => 'ssl-cert',
        key_contents         => "{{ key \"video/ssl_keys/${ingest_env}/wildcard_twitch_tv/key\" }}",
        key_restart_cmd      => $nginx_restart_command,
      }

      twitch_sandstorm_agent::template { 'ingest_twitch_tv_dhparams':
        destination     => '/etc/ssl/private/ingest_twitch_tv.dhparams',
        mode            => '0640',
        owner           => 'root',
        group           => 'ssl-cert',
        key             => "video/ssl_keys/${ingest_env}/ingest-proxy/dhparams",
        restart_command => $nginx_restart_command,
      }

      twitch_sandstorm_agent::template { 'ingest_twitch_tv_tls_key_current':
        destination     => '/etc/ssl/private/ingest_twitch_tv_tls_key_current',
        mode            => '0640',
        owner           => 'root',
        group           => 'ssl-cert',
        key             => "video/ssl_keys/${ingest_env}/ingest-proxy/tls_key_current",
        restart_command => $nginx_restart_command,
        trim_newline    => true,
        require         => Twitch_nginx::Vhost['ingestproxy'],
      }

      twitch_sandstorm_agent::template { 'ingest_twitch_tv_tls_key_previous':
        destination     => '/etc/ssl/private/ingest_twitch_tv_tls_key_previous',
        mode            => '0640',
        owner           => 'root',
        group           => 'ssl-cert',
        key             => "video/ssl_keys/${ingest_env}/ingest-proxy/tls_key_previous",
        restart_command => $nginx_restart_command,
        trim_newline    => true,
        require         => Twitch_nginx::Vhost['ingestproxy'],
      }

      twitch_sandstorm_agent::template { 'ingest_twitch_tv_tls_key_next':
        destination     => '/etc/ssl/private/ingest_twitch_tv_tls_key_next',
        mode            => '0640',
        owner           => 'root',
        group           => 'ssl-cert',
        key             => "video/ssl_keys/${ingest_env}/ingest-proxy/tls_key_next",
        restart_command => $nginx_restart_command,
        trim_newline    => true,
        require         => Twitch_nginx::Vhost['ingestproxy'],
      }

      # This installs Nagios passive checks onto servers that have secrets to monitor.
      realize(Twitch_nrdp::Check['check_sandstorm_agent','check_sandstorm_agent_ops'])
      twitch_nrdp::check { 'ingest_nginx_health':
        command  => '/usr/lib/nagios/plugins/check_http -I localhost -u "/health"',
        interval => 2,
        retry    => 4,
      }
    }
  }
}
