#
# default_gateway.rb
#
# This fact provides information about the default gateway (or the "route of
# last resort" if you wish) that is available on the system ...
#
require 'facter/util/file_read'

# hex2ip used to convert ip addresses from /proc/net/route
def hex2ip(h)
  if h.length == 8 then
	  "%d.%d.%d.%d" % ( h.scan(/[[:xdigit:]]{2,2}/).reverse.map {|x| x.to_i(16) } )
  else
    h
  end
end

if Facter.value(:kernel) == 'Linux'
  # We store information about the default gateway here ...
  gateway = ''
  gateway_interface = ''
  default_metric = 2147483648

  #
  # Modern Linux kernels provide "/proc/net/route" in the following format:
  #
  #  Iface      Destination     Gateway         Flags   RefCnt  Use     Metric  Mask            MTU     Window  IRTT
  #  eth0       0000FEA9        00000000        0001    0       0       1000    0000FFFF        0       0       0
  #  eth0       00006C0A        00000000        0001    0       0       1       0000FFFF        0       0       0
  #  eth0       00000000        460B6C0A        0003    0       0       0       00000000        0       0       0
  #

  # Perferred method vs cat https://www.rubydoc.info/gems/facter/Facter/Util/FileRead
  Facter::Util::FileRead.read('/proc/net/route').each_line do |line|
    # Remove bloat ...
    line.strip!

    # Skip header line ...
    next if line.match(/^[Ii]face.+/)

    # Skip new and empty lines ...
    next if line.match(/^(\r\n|\n|\s*)$|^$/)

    # Retrieve and set interface name, destination, gateway and metric
    interface_name,destination_hex,gateway_hex,metric = line.split("\t").values_at(0,1,2,6)

    next if destination_hex != '00000000' and gateway_hex == '00000000'

    # Skip if metric is larger
    next if metric.to_i > default_metric.to_i

    # Found a default route with a lower metric. lowest metric wins...
    default_metric = metric.to_i

    # Set interface name
    interface = interface_name

    # Set gateway ip
    gateway = hex2ip(gateway_hex)
    gateway_interface = interface

  end

  # Sometimes we do not have the default route set at all ...
  if gateway and not gateway.empty?
    Facter.add('default_gateway') do
      confine :kernel => :Linux
      setcode { gateway }
    end
  end
  if gateway_interface and not gateway_interface.empty?
    Facter.add('default_gateway_interface') do
      confine :kernel => :linux
      setcode { gateway_interface }
    end
  end
end

# vim: set ts=2 sw=2 et :
# encoding: utf-8
