# interface define
# used to manipulate /etc/network/interfaces.d/ and /var/run/network/
# Be careful passing ensure=absent & ipaddress='dhcp' the interface will come down.
#
define interfaces::interface (
  $ensure    = 'present',
  $ifname    = $title,
  $ipaddress = undef,
  $method    = 'dhcp',
  $netmask   = '255.255.255.255',
  $gateway   = undef,
  $auto      = false,
) {
  require ::interfaces

  validate_bool($auto)
  validate_slength($ifname, 15, 1)
  validate_re($ensure, '^(present|absent)$')
  validate_re($method, '^(dhcp|static)$')

  # This is to maintain backward compatibility with old bird::loopback define.
  $file_name = $ifname ? {
    /^lo:/  => "loopback_${ifname}",
    default => $ifname,
  }

  $family = is_ipv6_address($ipaddress) ? {
    true    => 'inet6',
    default => 'inet',
  }

  # this is really just an empty file, used by ifup/down utils
  file { "/var/run/network/ifup.${ifname}":
    ensure  => $ensure,
    content => '',
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    backup  => false,
  }

  if $ifname =~ /:/ {
    # custom resource to ensure we have our interface alias
    # listed in /var/run/network/ifstate
    ifstate { $ifname:
      ensure  => $ensure,
    }
    if defined(Exec["ifup ${ifname}"]) {
      Exec["ifup ${ifname}"] -> Ifstate[$ifname]
    }
  }
  case $ensure {
    'absent': {
      # The extra grep is because lo:anything brings up lo:* (all of them).
      $device_is_up = $ipaddress ? {
        'dhcp'  => "/sbin/ip addr show dev ${ifname} | /bin/grep -q ${ifname}",
        default => "/sbin/ip addr show dev ${ifname} | /bin/grep ${ifname} | /bin/grep -q ${ipaddress}",
      }
      exec { "/sbin/ifdown ${ifname}":
        onlyif => $device_is_up,
        before => [
          File["/etc/network/interfaces.d/${file_name}.cfg"],
          File["/var/run/network/ifup.${ifname}"],
        ],
      }
    }

    default: {
      if $method == 'dhcp' {
        # ifup will only run if interface is down.
        $interface_is_up = "/sbin/ip addr show dev ${ifname} | /bin/grep -q ${ifname}"
      } else {
        validate_ip_address("${ipaddress}/${netmask}")
        # ifup will only run if ip isnt listed on ANY interface
        $interface_is_up = "/sbin/ip addr list | /bin/grep -q ${ipaddress}"
      }

      if $auto {
        exec { "ifup ${title}":
          command => "/sbin/ifup ${ifname}",
          unless  => $interface_is_up,
          require => File["/etc/network/interfaces.d/${file_name}.cfg"],
          before  => File["/var/run/network/ifup.${ifname}"],
        }
      }
    }
  }

  file { "/etc/network/interfaces.d/${file_name}.cfg":
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template ("${module_name}/interface.cfg.erb"),
  }
}
